package storage

import (
	"bufio"
	"encoding/binary"
	"io"
	"os"
	"sync"
)

// OpType represents operation type
type OpType byte

const (
	OpPut    OpType = 1
	OpDelete OpType = 2
)

// WALEntry represents a write-ahead log entry
type WALEntry struct {
	Op    OpType
	Key   []byte
	Value []byte
}

// WriteAheadLog implements crash recovery
type WriteAheadLog struct {
	file   *os.File
	writer *bufio.Writer
	mu     sync.Mutex
}

// NewWriteAheadLog creates a new WAL
func NewWriteAheadLog(path string) (*WriteAheadLog, error) {
	file, err := os.OpenFile(path, os.O_CREATE|os.O_RDWR|os.O_APPEND, 0644)
	if err != nil {
		return nil, err
	}

	return &WriteAheadLog{
		file:   file,
		writer: bufio.NewWriter(file),
	}, nil
}

// Append writes an entry to WAL
func (wal *WriteAheadLog) Append(key, value []byte, op OpType) error {
	wal.mu.Lock()
	defer wal.mu.Unlock()

	// Format: [op:1][keyLen:4][key][valueLen:4][value]
	buf := make([]byte, 1+4+len(key)+4+len(value))
	offset := 0

	buf[offset] = byte(op)
	offset++

	binary.BigEndian.PutUint32(buf[offset:], uint32(len(key)))
	offset += 4

	copy(buf[offset:], key)
	offset += len(key)

	binary.BigEndian.PutUint32(buf[offset:], uint32(len(value)))
	offset += 4

	copy(buf[offset:], value)

	if _, err := wal.writer.Write(buf); err != nil {
		return err
	}

	return wal.writer.Flush()
}

// ReadAll reads all entries from WAL
func (wal *WriteAheadLog) ReadAll() ([]WALEntry, error) {
	wal.mu.Lock()
	defer wal.mu.Unlock()

	if _, err := wal.file.Seek(0, 0); err != nil {
		return nil, err
	}

	reader := bufio.NewReader(wal.file)
	entries := []WALEntry{}

	for {
		// Read op type
		opByte, err := reader.ReadByte()
		if err == io.EOF {
			break
		}
		if err != nil {
			return nil, err
		}

		// Read key length
		keyLenBuf := make([]byte, 4)
		if _, err := io.ReadFull(reader, keyLenBuf); err != nil {
			break
		}
		keyLen := binary.BigEndian.Uint32(keyLenBuf)

		// Read key
		key := make([]byte, keyLen)
		if _, err := io.ReadFull(reader, key); err != nil {
			break
		}

		// Read value length
		valueLenBuf := make([]byte, 4)
		if _, err := io.ReadFull(reader, valueLenBuf); err != nil {
			break
		}
		valueLen := binary.BigEndian.Uint32(valueLenBuf)

		// Read value
		value := make([]byte, valueLen)
		if _, err := io.ReadFull(reader, value); err != nil {
			break
		}

		entries = append(entries, WALEntry{
			Op:    OpType(opByte),
			Key:   key,
			Value: value,
		})
	}

	return entries, nil
}

// Reset clears the WAL
func (wal *WriteAheadLog) Reset() error {
	wal.mu.Lock()
	defer wal.mu.Unlock()

	if err := wal.file.Truncate(0); err != nil {
		return err
	}
	if _, err := wal.file.Seek(0, 0); err != nil {
		return err
	}

	wal.writer = bufio.NewWriter(wal.file)
	return nil
}

// Close closes the WAL
func (wal *WriteAheadLog) Close() error {
	wal.mu.Lock()
	defer wal.mu.Unlock()

	if err := wal.writer.Flush(); err != nil {
		return err
	}
	return wal.file.Close()
}
