# Complex Event Processing (CEP) Patterns Guide

A comprehensive guide to building, testing, and deploying CEP patterns in the Real-Time Analytics Engine.

## Table of Contents

1. [Pattern Fundamentals](#fundamentals)
2. [Pattern Definition](#definition)
3. [Common Patterns](#common-patterns)
4. [Building Custom Patterns](#custom)
5. [Testing Patterns](#testing)
6. [Real-World Examples](#examples)
7. [Performance Tuning](#performance)

## Fundamentals

### What is CEP?

Complex Event Processing detects meaningful patterns in event streams:

```
Raw Events: AAABBBAAACCCAAABBBCCC
Pattern:    Multiple As followed by Bs
Matches:    AAA → BBB ✓ | CCC ✗ | AAA → BBB ✓
```

### CEP vs Simple Filtering

**Simple Filter** (Count events):
```go
count := 0
for event := range events {
    if event.Amount > 100 {
        count++
    }
}
// Result: 500 transactions > $100
```

**CEP Pattern** (Detect behavior):
```go
pattern := {
    "5 purchases > $100",
    "from different cities",
    "within 1 hour",
}
// Result: Fraud detected at time 10:30:45
```

### CEP Use Cases

| Use Case | Pattern | Trigger |
|----------|---------|---------|
| Fraud Detection | Multi-location purchases | 5+ cities in 1 hour |
| System Monitoring | Error rate spike | 10+ errors in 1 min |
| IoT Alerts | Temperature surge | Temp > 80°C for 5+ readings |
| Trading | Trend reversal | Price crosses 2 moving averages |
| Security | DDoS attack | 1000+ requests from single IP |

## Pattern Definition

### Pattern Structure

```go
type Pattern struct {
    // Identification
    Name        string

    // What to detect
    Conditions  []Condition

    // Time constraint
    TimeWindow  time.Duration

    // What to do
    Actions     []Action
}
```

### Conditions

```go
type Condition struct {
    // Filter by event type
    EventType   string

    // Custom filter logic
    Predicate   func(*Event) bool

    // Occurrence constraints
    MinCount    int  // At least N occurrences
    MaxCount    int  // At most N occurrences (0 = unlimited)
}
```

**Examples**:

```go
// Simple condition
Condition{
    EventType: "purchase",
    Predicate: func(e *stream.Event) bool {
        return e.Value["amount"].(float64) > 100
    },
    MinCount: 5,
}

// Complex predicate
Condition{
    EventType: "login",
    Predicate: func(e *stream.Event) bool {
        location := e.Value["location"].(string)
        country := e.Value["country"].(string)
        return location != "home" && country != "US"
    },
    MinCount: 3,
}

// Temporal condition
Condition{
    EventType: "request",
    Predicate: func(e *stream.Event) bool {
        latency := e.Value["latency_ms"].(float64)
        return latency > 5000  // 5 seconds
    },
    MinCount: 10,  // 10 slow requests
}
```

### Actions

Actions execute when pattern matches:

```go
type Action interface {
    Execute(matches []*Event)
}

// Alert action
type AlertAction struct {
    AlertType string  // "fraud", "security", etc.
    Severity  string  // "critical", "high", "medium", "low"
    Channel   string  // "email", "slack", "pagerduty"
}

// Logging action
type LogAction struct {
    Level   string  // "error", "warning", "info"
}

// Webhook action
type WebhookAction struct {
    URL string
}
```

## Common Patterns

### 1. Sequential Patterns

**Detect**: Event A followed by Event B

```go
Pattern{
    Name: "Purchase then Chargeback",
    Conditions: []Condition{
        {
            EventType: "purchase",
            Predicate: func(e *stream.Event) bool {
                return e.Value["amount"].(float64) > 50
            },
            MinCount: 1,
        },
        {
            EventType: "chargeback",
            Predicate: func(e *stream.Event) bool {
                return true
            },
            MinCount: 1,
        },
    },
    TimeWindow: 30 * 24 * time.Hour,  // 30 days
    Actions: []Action{
        &AlertAction{Severity: "high"},
    },
}
```

### 2. Repetition Patterns

**Detect**: Same event type N times

```go
Pattern{
    Name: "Multiple Failed Logins",
    Conditions: []Condition{
        {
            EventType: "login_failed",
            Predicate: func(e *stream.Event) bool {
                return true
            },
            MinCount: 5,  // 5 failures minimum
            MaxCount: 0,  // Unlimited maximum
        },
    },
    TimeWindow: 10 * time.Minute,
    Actions: []Action{
        &AlertAction{
            AlertType: "security",
            Severity:  "critical",
        },
    },
}
```

### 3. Absence Patterns

**Detect**: Missing expected event

```go
Pattern{
    Name: "Session Without Logout",
    Conditions: []Condition{
        {
            EventType: "login",
            Predicate: func(e *stream.Event) bool {
                return true
            },
            MinCount: 1,
        },
        // Note: Would need custom absence check
        // (Requires pattern matcher enhancement)
    },
    TimeWindow: 24 * time.Hour,
    Actions: []Action{
        &AlertAction{Severity: "low"},
    },
}
```

### 4. Aggregation Patterns

**Detect**: Multiple events meeting aggregate condition

```go
Pattern{
    Name: "High Daily Spending",
    Conditions: []Condition{
        {
            EventType: "purchase",
            Predicate: func(e *stream.Event) bool {
                return e.Value["amount"].(float64) > 10
            },
            MinCount: 10,  // 10+ purchases
        },
    },
    TimeWindow: 24 * time.Hour,
    Actions: []Action{
        &AlertAction{Severity: "medium"},
    },
}
```

### 5. Outlier Patterns

**Detect**: Event significantly different from baseline

```go
Pattern{
    Name: "Unusual Request Size",
    Conditions: []Condition{
        {
            EventType: "api_request",
            Predicate: func(e *stream.Event) bool {
                payloadSize := e.Value["payload_bytes"].(float64)
                return payloadSize > 1000000  // 1MB (unusual)
            },
            MinCount: 3,  // Multiple large requests
        },
    },
    TimeWindow: 5 * time.Minute,
    Actions: []Action{
        &AlertAction{Severity: "high"},
    },
}
```

## Custom Patterns

### Building a Real-World Pattern

**Scenario**: Detect account takeover attempts

**Requirements**:
1. Multiple failed login attempts (5+)
2. Successful login from unusual location
3. Quick user data modification
4. All within 15 minutes

**Implementation**:

```go
type AccountTakeoverPattern struct {
    UserID string
    StartTime time.Time
    FailedLogins int
    SuspiciousLocations []string
}

Pattern{
    Name: "Account Takeover Detection",
    Conditions: []Condition{
        // Condition 1: Failed login attempts
        {
            EventType: "login_failed",
            Predicate: func(e *stream.Event) bool {
                return e.Value["reason"].(string) == "invalid_password"
            },
            MinCount: 5,
        },
        // Condition 2: Successful login
        {
            EventType: "login_success",
            Predicate: func(e *stream.Event) bool {
                location := e.Value["location"].(string)
                // Identify unusual locations
                return !isKnownLocation(location)
            },
            MinCount: 1,
        },
        // Condition 3: Quick data modification
        {
            EventType: "user_profile_updated",
            Predicate: func(e *stream.Event) bool {
                return e.Value["field"].(string) == "phone" ||
                       e.Value["field"].(string) == "email"
            },
            MinCount: 1,
        },
    },
    TimeWindow: 15 * time.Minute,
    Actions: []Action{
        &AlertAction{
            AlertType: "security",
            Severity:  "critical",
            Channel:   "pagerduty",
        },
    },
}

func isKnownLocation(location string) bool {
    knownLocations := map[string]bool{
        "office":   true,
        "home":     true,
        "commute":  true,
    }
    return knownLocations[location]
}
```

### Custom State in Patterns

For complex patterns, extend pattern matching:

```go
type AdvancedPatternMatcher struct {
    pattern         *Pattern
    partialMatches  map[string]*AdvancedMatch
    userProfiles    map[string]*UserProfile  // Custom state
    anomalies       map[string]*Anomaly
    mu              sync.RWMutex
}

type UserProfile struct {
    UserID              string
    UsualLocations      []string
    AverageSpending     float64
    PeakActivityHours   []int
}

type Anomaly struct {
    DetectedAt time.Time
    Type       string  // "location", "spending", "time"
    Severity   float64 // 0.0-1.0
}

func (apm *AdvancedPatternMatcher) Process(event *stream.Event) {
    apm.mu.Lock()
    defer apm.mu.Unlock()

    userID := event.UserID
    profile := apm.getUserProfile(userID)

    // Detect anomalies
    if apm.isAnomalousLocation(event, profile) {
        apm.recordAnomaly(userID, "location", 0.8)
    }

    if apm.isAnomalousSpending(event, profile) {
        apm.recordAnomaly(userID, "spending", 0.9)
    }

    // Check pattern with anomaly context
    apm.checkPattern(event, profile)
}

func (apm *AdvancedPatternMatcher) isAnomalousLocation(
    event *stream.Event,
    profile *UserProfile,
) bool {
    location := event.Value["location"].(string)

    for _, knownLoc := range profile.UsualLocations {
        if location == knownLoc {
            return false
        }
    }

    return true
}
```

## Testing Patterns

### Unit Tests

```go
func TestMultiCityFraudPattern(t *testing.T) {
    pattern := &cep.Pattern{
        Name: "Multi-City Fraud",
        Conditions: []cep.Condition{
            {
                EventType: "purchase",
                Predicate: func(e *stream.Event) bool {
                    amount := e.Value["amount"].(float64)
                    return amount > 100
                },
                MinCount: 5,
            },
        },
        TimeWindow: 1 * time.Hour,
    }

    matcher := cep.NewPatternMatcher(pattern)

    // Create test events
    baseTime := time.Now()
    events := []*stream.Event{
        {
            Timestamp: baseTime.Add(0 * time.Second),
            EventType: "purchase",
            UserID:    9999,
            Value: map[string]interface{}{
                "amount": 250.0,
                "city":   "New York",
            },
        },
        {
            Timestamp: baseTime.Add(1 * time.Minute),
            EventType: "purchase",
            UserID:    9999,
            Value: map[string]interface{}{
                "amount": 300.0,
                "city":   "Los Angeles",
            },
        },
        // ... more events ...
    }

    // Process and verify
    for _, event := range events {
        matcher.Process(event)
    }

    // Assert pattern matched
    // (Would require adding result checking to matcher)
}
```

### Integration Tests

```go
func TestPatternInFullPipeline(t *testing.T) {
    // Setup
    operator := stream.NewWindowOperator(
        stream.WindowTypeTumbling,
        1*time.Minute,
        0, 0,
        &stream.CountAggregator{},
    )

    matcher := cep.NewPatternMatcher(fraudPattern)

    // Generate test data
    testData := generateTestEvents(1000)

    // Process through pipeline
    for _, event := range testData {
        // Window processing
        operator.Process(event)

        // Pattern matching
        matcher.Process(event)
    }

    // Verify expected patterns detected
    // Verify no false positives
}
```

### Simulating Pattern Scenarios

```bash
#!/bin/bash
# Test fraud detection pattern

# Simulate clean purchases (no fraud)
for i in {1..5}; do
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"purchase\",
      \"user_id\": 1001,
      \"key\": \"user-1001\",
      \"value\": {
        \"amount\": 50,
        \"city\": \"New York\"
      }
    }"
  sleep 1
done

# Simulate fraud (multiple cities, high amount)
declare -a cities=("New York" "Los Angeles" "Chicago" "Miami" "Seattle")
for city in "${cities[@]}"; do
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"purchase\",
      \"user_id\": 9999,
      \"key\": \"user-9999\",
      \"value\": {
        \"amount\": 250,
        \"city\": \"$city\"
      }
    }"
  sleep 2
done

echo "Check analytics engine logs for pattern matches"
```

## Real-World Examples

### Example 1: E-Commerce Fraud Detection

**Pattern**: Detect stolen credit cards

```go
stolenCardPattern := &cep.Pattern{
    Name: "Stolen Credit Card Detection",
    Conditions: []cep.Condition{
        {
            EventType: "failed_payment",
            Predicate: func(e *stream.Event) bool {
                reason := e.Value["decline_reason"].(string)
                return reason == "invalid_card" ||
                       reason == "suspected_fraud"
            },
            MinCount: 3,  // 3+ failed attempts
        },
        {
            EventType: "successful_payment",
            Predicate: func(e *stream.Event) bool {
                // Successful after failures = suspicious
                return true
            },
            MinCount: 1,
        },
        {
            EventType: "large_purchase",
            Predicate: func(e *stream.Event) bool {
                amount := e.Value["amount"].(float64)
                return amount > 1000  // Large purchase after failures
            },
            MinCount: 1,
        },
    },
    TimeWindow: 1 * time.Hour,
    Actions: []Action{
        &AlertAction{
            AlertType: "fraud",
            Severity:  "critical",
            Channel:   "pagerduty",
        },
    },
}
```

### Example 2: System Performance Degradation

**Pattern**: Detect cascading failures

```go
cascadingFailurePattern := &cep.Pattern{
    Name: "Cascading Service Failure",
    Conditions: []cep.Condition{
        {
            EventType: "service_error",
            Predicate: func(e *stream.Event) bool {
                // Increasing error rate
                errorRate := e.Value["error_rate"].(float64)
                return errorRate > 0.1  // >10% error rate
            },
            MinCount: 5,  // 5 services affected
        },
        {
            EventType: "response_timeout",
            Predicate: func(e *stream.Event) bool {
                return true
            },
            MinCount: 10,  // Multiple timeouts
        },
    },
    TimeWindow: 5 * time.Minute,
    Actions: []Action{
        &AlertAction{
            Severity: "critical",
            Channel:  "pagerduty",
        },
    },
}
```

### Example 3: Regulatory Compliance

**Pattern**: Detect suspicious trading activity

```go
suspiciousTradingPattern := &cep.Pattern{
    Name: "Suspicious Trading Pattern",
    Conditions: []cep.Condition{
        {
            EventType: "buy_order",
            Predicate: func(e *stream.Event) bool {
                // Large order
                quantity := e.Value["quantity"].(float64)
                return quantity > 100000
            },
            MinCount: 3,
        },
        {
            EventType: "sell_order",
            Predicate: func(e *stream.Event) bool {
                // Immediate sell
                return true
            },
            MinCount: 3,
        },
    },
    TimeWindow: 5 * time.Minute,
    Actions: []Action{
        &AlertAction{
            AlertType: "compliance",
            Severity:  "high",
        },
    },
}
```

## Performance Tuning

### Optimize Predicate Logic

**Inefficient**:
```go
Predicate: func(e *stream.Event) bool {
    // Multiple type assertions
    if amount, ok := e.Value["amount"].(float64); ok {
        if amount > 100 {
            if city, ok := e.Value["city"].(string); ok {
                // ... more checks ...
            }
        }
    }
    return false
}
```

**Optimized**:
```go
Predicate: func(e *stream.Event) bool {
    // Early returns, short-circuit evaluation
    amount, ok := e.Value["amount"].(float64)
    if !ok || amount <= 100 {
        return false
    }

    city, ok := e.Value["city"].(string)
    if !ok || len(city) == 0 {
        return false
    }

    return true
}
```

### Partition Complex Patterns

**Instead of single large pattern**:
```go
// Bad: One pattern with 10 conditions
pattern := Pattern{
    Conditions: []Condition{ ... 10 conditions ... },
}
```

**Do: Multiple simpler patterns**:
```go
// Good: Cascade simpler patterns
simplePattern1 := Pattern{
    Conditions: []Condition{ ... 3 conditions ... },
}

simplePattern2 := Pattern{
    Conditions: []Condition{ ... 3 conditions ... },
}

simplePattern3 := Pattern{
    Conditions: []Condition{ ... 3 conditions ... },
}

// Feed output of pattern1 into pattern2, etc.
```

### Pre-Filter Events

```go
// Create filter matcher before CEP
type FilteredMatcher struct {
    preFilter  func(*stream.Event) bool
    matcher    *cep.PatternMatcher
}

func (fm *FilteredMatcher) Process(event *stream.Event) {
    // Skip obvious non-matches early
    if !fm.preFilter(event) {
        return
    }

    // Only pass relevant events to expensive pattern matcher
    fm.matcher.Process(event)
}
```

### Memory Management

```go
// Limit partial match storage
type LimitedPatternMatcher struct {
    pattern         *Pattern
    partialMatches  map[string]*PartialMatch
    maxMatches      int  // Limit to prevent memory growth
    mu              sync.RWMutex
}

func (pm *LimitedPatternMatcher) Process(event *stream.Event) {
    pm.mu.Lock()
    defer pm.mu.Unlock()

    // Aggressive cleanup
    pm.cleanupExpired()

    // Limit growth
    if len(pm.partialMatches) > pm.maxMatches {
        pm.purgeOldest()
    }

    // ... normal processing ...
}
```

---

**CEP Expert Ready!**

Start detecting patterns and securing your data streams.
