# Real-Time Analytics Engine - Complete Documentation Index

Welcome to the Real-Time Analytics Engine documentation. This is your guide to understanding and using the platform.

## Quick Navigation

### For Getting Started
- **[README.md](README.md)** - Feature overview and quick start (5 min read)
- **[README-ENHANCED.md](README-ENHANCED.md)** - Comprehensive guide with all details (30 min read)

### For Understanding the System
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - System design and components
- **[STREAM_PROCESSING_GUIDE.md](STREAM_PROCESSING_GUIDE.md)** - Stream processing concepts and windowing

### For Advanced Features
- **[CEP_PATTERNS_GUIDE.md](CEP_PATTERNS_GUIDE.md)** - Pattern matching, fraud detection, anomalies
- **[examples/queries.md](examples/queries.md)** - API examples, use cases, code snippets

### For Development
- **[Makefile](Makefile)** - Build commands and development tools
- **[docker-compose.yml](docker-compose.yml)** - Local development setup

## Documentation Organization

### 📋 Core Documentation (Read First)

1. **README.md** (3 pages)
   - Project overview
   - Key features (stream processing, windowing, CEP)
   - Quick start instructions
   - Basic usage examples
   - Performance benchmarks

2. **README-ENHANCED.md** (50 pages)
   - Complete feature breakdown
   - Architecture diagrams
   - API reference
   - Configuration guide
   - Detailed stream processing examples
   - CEP pattern examples
   - Horizontal scaling guide
   - Production deployment
   - Troubleshooting guide

### 🏗️ Architecture & Design

3. **ARCHITECTURE.md** (5 pages)
   - System overview
   - Core components
   - Data flow diagrams
   - Concurrency model
   - Performance characteristics
   - Scalability approach
   - Fault tolerance
   - Monitoring

### 📊 Stream Processing

4. **STREAM_PROCESSING_GUIDE.md** (25 pages)
   - Fundamental concepts
   - Tumbling windows (fixed-size)
   - Sliding windows (overlapping)
   - Session windows (activity-based)
   - Watermarks and late events
   - Aggregation patterns
   - State management
   - Memory optimization
   - Exactly-once semantics

### 🔍 Complex Event Processing

5. **CEP_PATTERNS_GUIDE.md** (20 pages)
   - Pattern fundamentals
   - Pattern definition
   - Common pattern types
   - Custom patterns
   - Testing patterns
   - Real-world examples (fraud, DDoS, trading)
   - Performance tuning

### 💡 Examples & Queries

6. **examples/queries.md** (20 pages)
   - API examples
   - Use case scenarios
   - Performance testing
   - Python and Go clients
   - Monitoring queries

## Learning Paths

### Path 1: Get Running in 10 Minutes
1. Read: README.md (sections: Overview, Quick Start, Usage)
2. Run: `make demo`
3. Open: http://localhost:8080
4. Try: Send events via curl (see examples/queries.md)

### Path 2: Understand Stream Processing (30 minutes)
1. Read: README-ENHANCED.md sections:
   - Stream Processing
   - Window Types
   - Aggregations
2. Study: STREAM_PROCESSING_GUIDE.md sections:
   - Fundamentals
   - Window Types Deep Dive
   - Watermarks

### Path 3: Implement Fraud Detection (1 hour)
1. Read: README-ENHANCED.md → Complex Event Processing
2. Study: CEP_PATTERNS_GUIDE.md → Examples → Fraud Detection
3. Review: examples/queries.md → Use Case 1: E-commerce Fraud
4. Code: Implement custom pattern matching

### Path 4: Deploy to Production (2 hours)
1. Read: README-ENHANCED.md → Production Deployment
2. Review: ARCHITECTURE.md → Scalability & Fault Tolerance
3. Study: docker-compose.yml and Dockerfile
4. Plan: Kubernetes deployment (YAML examples in README)

### Path 5: Master Stream Processing (3 hours)
1. Study: STREAM_PROCESSING_GUIDE.md (entire)
2. Study: ARCHITECTURE.md (concurrency model)
3. Review: Source code (pkg/stream/)
4. Implement: Custom aggregators and window operations

## Feature Directory

### Stream Processing Features
- **Tumbling Windows** → STREAM_PROCESSING_GUIDE.md § Tumbling Windows
- **Sliding Windows** → STREAM_PROCESSING_GUIDE.md § Sliding Windows
- **Session Windows** → STREAM_PROCESSING_GUIDE.md § Session Windows
- **Watermarks** → STREAM_PROCESSING_GUIDE.md § Watermarks and Late Events
- **Aggregations** → README-ENHANCED.md § Aggregation Functions

### Complex Event Processing
- **Pattern Matching** → CEP_PATTERNS_GUIDE.md § Pattern Fundamentals
- **Fraud Detection** → CEP_PATTERNS_GUIDE.md § Example 1: E-Commerce Fraud
- **System Monitoring** → CEP_PATTERNS_GUIDE.md § Example 2: System Performance
- **Custom Patterns** → CEP_PATTERNS_GUIDE.md § Building Custom Patterns

### Infrastructure & Deployment
- **Docker Setup** → docker-compose.yml
- **Kubernetes** → README-ENHANCED.md § Production Deployment
- **Configuration** → README-ENHANCED.md § Configuration
- **Scaling** → README-ENHANCED.md § Horizontal Scaling

### Development & Testing
- **Build Commands** → Makefile
- **Testing** → README.md § Testing
- **Benchmarks** → README-ENHANCED.md § Performance Benchmarks
- **Examples** → examples/queries.md

## Source Code Map

### Core Stream Processing
```
pkg/stream/
├── event.go              # Event model, window types
├── windowing.go          # WindowOperator, window assignment
├── aggregators.go        # COUNT, SUM, AVG, MIN, MAX, PERCENTILE
├── aggregators_test.go   # Unit tests for aggregators
└── windowing_test.go     # Unit tests for windowing
```

### Complex Event Processing
```
pkg/cep/
└── pattern.go            # Pattern matching, condition evaluation
```

### Real-Time Updates
```
pkg/websocket/
└── server.go             # WebSocket server, Pub/Sub, broadcasting
```

### Storage & Persistence
```
pkg/storage/
└── clickhouse.go         # Time-series storage with ClickHouse
```

### Application Server
```
cmd/server/
└── main.go               # HTTP API, configuration, integration
```

## Command Reference

### Development
```bash
make demo              # Run with demo data (no external deps)
make build             # Build binary
make run               # Build and run
make test              # Run unit tests
make test-coverage     # Generate coverage report
make bench             # Run benchmarks
```

### Code Quality
```bash
make fmt               # Format code
make vet               # Run go vet
make lint              # Run linter
```

### Docker & Deployment
```bash
make docker-build      # Build Docker image
make docker-up         # Start Docker Compose
make docker-down       # Stop Docker Compose
make docker-logs       # View logs
```

## API Quick Reference

### Event Ingestion
```bash
POST /api/ingest
Content-Type: application/json

{
  "timestamp": "2024-01-15T10:30:00Z",
  "event_type": "purchase",
  "user_id": 12345,
  "key": "user-12345",
  "value": { "amount": 199.99, "city": "New York" }
}
```

### Historical Query
```bash
GET /api/query?type=purchase&start=2024-01-15T09:00:00Z&end=2024-01-15T10:00:00Z
```

### Statistics
```bash
GET /api/stats
```

### WebSocket Connection
```
ws://localhost:8081/ws

Subscribe message:
{
  "action": "subscribe",
  "topics": ["dashboard:*"]
}
```

## Glossary of Terms

### Stream Processing
- **Event**: Single data point (purchase, login, error)
- **Stream**: Unbounded sequence of events
- **Window**: Time-based grouping of events
- **Watermark**: Threshold for handling late events
- **Aggregation**: Combining events (COUNT, SUM, AVG, etc.)
- **Partition**: Dividing stream across workers by key

### Complex Event Processing
- **Pattern**: Rule for detecting event sequences
- **Condition**: Filter/predicate for events
- **Action**: Callback when pattern matches
- **Partial Match**: In-progress pattern detection

### Storage
- **Columnar**: Data organized by column (ClickHouse)
- **Time-Series**: Data indexed by timestamp
- **Batch Insert**: Multiple rows inserted together
- **Checkpoint**: Saved state for recovery

### System
- **Throughput**: Events processed per second
- **Latency**: Time from event to result
- **Partition**: Horizontal scaling unit
- **Idempotent**: Safe to process multiple times

## Common Questions

**Q: How do I ingest data?**
A: Use HTTP POST to `/api/ingest` with JSON events. See examples/queries.md

**Q: What's the throughput?**
A: 1M+ events/sec per node. See STREAM_PROCESSING_GUIDE.md for details.

**Q: How do I detect fraud?**
A: Use CEP patterns. See CEP_PATTERNS_GUIDE.md § Fraud Detection

**Q: Can I run without ClickHouse?**
A: Yes, uses in-memory storage. Perfect for development.

**Q: How do I scale horizontally?**
A: Partition by event key across workers. See README-ENHANCED.md § Horizontal Scaling

**Q: How do I deploy to Kubernetes?**
A: See YAML config in README-ENHANCED.md § Production Deployment

## Resources

### Papers & Research
- [The Dataflow Model (Google)](https://research.google/pubs/pub43864/)
- [Apache Flink Architecture](https://flink.apache.org/)
- [Stream Processing with Apache Kafka Streams](https://kafka.apache.org/documentation/streams/)

### Tools & Libraries
- [Gorilla WebSocket](https://github.com/gorilla/websocket)
- [Redis Go Client](https://github.com/redis/go-redis)
- [ClickHouse Go Driver](https://github.com/ClickHouse/clickhouse-go)

### Related Technologies
- Kafka (event streaming)
- Redis (Pub/Sub, caching)
- ClickHouse (time-series DB)
- Kubernetes (orchestration)

## Getting Help

### Documentation
1. Check the relevant section above
2. Search documentation for keywords
3. Review examples in examples/queries.md

### Troubleshooting
- See README-ENHANCED.md § Troubleshooting
- Check logs: `docker-compose logs analytics-engine`
- Monitor metrics: `curl http://localhost:8080/api/stats`

### Community
- Open issues on GitHub
- Check existing issues for solutions
- Join discussions and chat

## File Statistics

- **Total Documentation Pages**: 100+
- **Code Examples**: 200+
- **Use Cases Covered**: 10+
- **API Endpoints**: 4 primary + WebSocket
- **Window Types**: 3 (tumbling, sliding, session)
- **Aggregation Functions**: 6+ built-in
- **Pattern Examples**: 10+

## Version Information

**Documentation Version**: 1.0.0
**Last Updated**: 2024-10-28
**Engine Version**: 1.0.0
**Go Version**: 1.21+

---

**Happy Building!**

Start with README.md or jump to your specific learning path above.
