# Stream Processing Guide

A comprehensive guide to stream processing concepts and patterns in the Real-Time Analytics Engine.

## Table of Contents

1. [Stream Processing Fundamentals](#fundamentals)
2. [Window Types Deep Dive](#windows)
3. [Watermarks and Late Events](#watermarks)
4. [Aggregation Patterns](#aggregations)
5. [Partitioning and State Management](#state)
6. [Complex Event Processing Patterns](#cep-patterns)
7. [Performance Optimization](#optimization)
8. [Exactly-Once Semantics](#exactly-once)

## Fundamentals

### Stream Processing Model

Stream processing transforms unbounded data streams into actionable insights:

```
Data Source → Stream → Processing → Output Sink
   ▲                        ▲
   │                        │
  Live,                   Real-time
Continuous              Analytics
```

### Key Properties

**Unbounded Data**: Streams never end
```
Time: |--E1--E2---------E3--E4--E5-----E6------->
      (Events arrive unpredictably)
```

**Latency Requirements**: Results needed quickly
```
Event → Processing → Result
  |                    ▲
  |____________________|
  (Should be < 100ms)
```

**Fault Tolerance**: Must survive failures
```
If Worker crashes → Results reproducible from state
```

## Window Types

### Tumbling Windows (Fixed-Size, Non-Overlapping)

**Characteristics**:
- Fixed duration (e.g., 1 minute)
- Non-overlapping (each event in exactly 1 window)
- Natural boundaries aligned to time

**Visual**:
```
Events:  │ A  B  C │ D  E  F │ G  H  I │
Time:    │ 0:00-01 │ 0:01-02 │ 0:02-03 │
Window:  │ Window1 │ Window2 │ Window3 │
```

**Use Cases**:
- Event counts per minute
- Hourly revenue summaries
- Daily active users
- Minute-by-minute metrics

**Implementation**:
```go
tumbleOp := stream.NewWindowOperator(
    stream.WindowTypeTumbling,
    1*time.Minute,  // 1-minute windows
    0, 0,           // Not used for tumbling
    &stream.CountAggregator{},
)

// Each event assigned to exactly one window
for event := range events {
    results := tumbleOp.Process(event)
    // Results emitted when window closes
}
```

**State Management**:
```
Memory active: Only 2 windows at most (current + next)
Storage: Continuous, no overlap
Example: 1000 events/minute = 1000 in-memory entries per window
```

### Sliding Windows (Overlapping)

**Characteristics**:
- Window size (e.g., 5 minutes)
- Slide interval (e.g., 1 minute)
- Windows overlap significantly

**Visual**:
```
Time:     0   1   2   3   4   5   6   7   8
          │───────────────│  (5-min window)
          │   │───────────────│  (slide by 1)
          │   │   │───────────────│
Events:   A B C D E F G H I J
```

**Overlap Count**:
```
Window size / Slide interval = 5 min / 1 min = 5 overlapping windows
(Each event contributes to 5 different windows)
```

**Use Cases**:
- Moving averages (5-minute sliding latency)
- Trend detection
- Real-time dashboards (update every N seconds)
- Anomaly detection with time context

**Implementation**:
```go
slideOp := stream.NewWindowOperator(
    stream.WindowTypeSliding,
    5*time.Minute,  // Window size
    1*time.Minute,  // Slide interval
    0,              // Not used
    &stream.AvgAggregator{Field: "latency"},
)

// Results emitted every slide interval
for event := range events {
    results := slideOp.Process(event)
    // 5 windows updated, oldest potentially emitted
}
```

**State Complexity**:
```
Memory active: Size/Slide = 5 windows at most
Storage growth: Much larger than tumbling
Example: 1000 events/minute = 5000 in-memory entries
         (because each event in 5 windows)
```

### Session Windows (Activity-Based)

**Characteristics**:
- Defined by inactivity gap (e.g., 30 seconds)
- Dynamic window boundaries
- Windows merge on activity continuation

**Visual**:
```
Events:   A B C D E    gap    F G H I    gap    J K L
Time:     │─────────│         │─────────│         │────│
Window:   │ Session1 │        │ Session2 │        │Sess│

(If gap extends beyond 30s, windows close)
```

**Merge Behavior**:
```
Activity sequence: E1 -- E2 -- (long gap) -- E3 -- E4
                   └─────────────────┬────────────────┘
                         Two sessions (gap exceeded)
```

**Use Cases**:
- User session analysis
- IoT sensor clustering
- Application request grouping
- Event correlation by time proximity

**Implementation**:
```go
sessionOp := stream.NewWindowOperator(
    stream.WindowTypeSession,
    0,              // Not used
    0,              // Not used
    30*time.Second, // Inactivity gap
    &stream.AvgAggregator{Field: "temperature"},
)

// Window automatically closes after 30s of inactivity
for event := range events {
    results := sessionOp.Process(event)
    // Results emitted when gap exceeded
}
```

**State Management**:
```
Memory active: Dynamic (1 window per active session/key)
Example: 10k concurrent users = ~10k active windows
         (Each storing partial aggregation)
```

### Window Comparison

| Aspect | Tumbling | Sliding | Session |
|--------|----------|---------|---------|
| Overlap | None | Heavy | None |
| Boundaries | Time-based | Time-based | Event-based |
| State Size | Low | Medium-High | Variable |
| Complexity | Simple | Medium | Complex |
| Latency | Fixed | Frequent updates | Variable |
| Use Case | Metrics | Trends | Sessions |

## Watermarks and Late Events

### Watermark Concept

Watermarks solve the problem of **late and out-of-order events**.

**Problem**:
```
Expected order: E1 → E2 → E3 → E4
Actual arrival: E1 → E3 → E2 → E4

When should window close? How to handle E2?
```

**Solution**:
```
Watermark = Maximum event timestamp seen - Allowed lateness

Events before watermark: Process normally
Events equal/after watermark: Drop as too late
```

### Watermark Propagation

```
Events:      │ 10:00:10 │ 10:00:15 │ 10:00:08 │ 10:00:20 │
Time:        10:00:00  10:00:01  10:00:02  10:00:03

Watermark*:  0         5s        8s        15s
             (max - 5s lateness allowed)

Processing:  ✓         ✓         ✓ (within lateness)
Status:      Assigned  Assigned  Assigned
```

### Late Event Handling

**Configuration**:
```go
// Allow 5 seconds of lateness
watermark := event.Timestamp.Add(-5 * time.Second)
```

**Example Scenario**:
```
Watermark: 10:00:25

Incoming event: 10:00:20 (5 seconds late)
Result: PROCESS ✓ (within allowed lateness)

Incoming event: 10:00:18 (7 seconds late)
Result: DROP ✗ (exceeds allowed lateness)
```

### Implementing Watermark Logic

```go
func (wo *WindowOperator) Process(event *Event) []WindowResult {
    // Update watermark
    if event.Timestamp.After(wo.watermark) {
        newWatermark := event.Timestamp.Add(-5 * time.Second)

        if newWatermark.After(wo.watermark) {
            wo.watermark = newWatermark

            // Close windows older than watermark
            for key, state := range wo.windows {
                if state.window.IsExpired(wo.watermark) {
                    // Emit window result
                    result := wo.emitWindow(state)
                    delete(wo.windows, key)
                }
            }
        }
    }

    // Process event if not too late
    if event.Timestamp.Before(wo.watermark.Add(5 * time.Second)) {
        // Assign to windows
        return wo.assignWindows(event)
    }

    // Event too late, drop it
    return []WindowResult{}
}
```

## Aggregation Patterns

### Aggregate State Representation

**Stateful Aggregation**:
```go
type State struct {
    count int
    sum   float64
    min   float64
    max   float64
    // ... other fields
}
```

**Streaming Update**:
```
Event 1: count=1, sum=10, min=10, max=10
Event 2: count=2, sum=25, min=10, max=15
Event 3: count=3, sum=48, min=10, max=23
```

### Single-Pass Aggregations

**Algorithms that work in one pass**:

**COUNT**:
```go
func (ca *CountAggregator) Add(current interface{}, event *Event) interface{} {
    if current == nil {
        return 1
    }
    return current.(int) + 1
}
```

**SUM/AVG** (with count):
```go
type AvgState struct {
    sum   float64
    count int
}

func (aa *AvgAggregator) Add(current interface{}, event *Event) interface{} {
    state := current.(*AvgState)
    if state == nil {
        state = &AvgState{}
    }

    value := event.Value[aa.Field].(float64)
    state.sum += value
    state.count++

    return state
}

func (aa *AvgAggregator) Result(aggregate interface{}) interface{} {
    state := aggregate.(*AvgState)
    if state.count == 0 {
        return 0.0
    }
    return state.sum / float64(state.count)
}
```

**MIN/MAX**:
```go
type MinMaxState struct {
    min float64
    max float64
    count int
}

func (mma *MinAggregator) Add(current interface{}, event *Event) interface{} {
    state := current.(*MinMaxState)
    if state == nil {
        state = &MinMaxState{
            min: math.MaxFloat64,
            max: -math.MaxFloat64,
        }
    }

    value := event.Value[mma.Field].(float64)
    if value < state.min {
        state.min = value
    }
    if value > state.max {
        state.max = value
    }
    state.count++

    return state
}
```

### Approximate Aggregations

**PERCENTILE** (using streaming algorithm):
```go
type PercentileState struct {
    samples []float64  // Sample values
    maxSize int
}

func (pa *PercentileAggregator) Add(current interface{}, event *Event) interface{} {
    state := current.(*PercentileState)
    if state == nil {
        state = &PercentileState{
            samples: []float64{},
            maxSize: 10000,  // Sample 10k values
        }
    }

    value := event.Value[pa.Field].(float64)

    if len(state.samples) < state.maxSize {
        // Reservoir sampling
        state.samples = append(state.samples, value)
    } else {
        // Randomly replace with decreasing probability
        idx := rand.Intn(len(state.samples))
        state.samples[idx] = value
    }

    return state
}

func (pa *PercentileAggregator) Result(aggregate interface{}) interface{} {
    state := aggregate.(*PercentileState)

    // Sort samples
    sort.Float64s(state.samples)

    // Calculate percentile
    idx := int(float64(len(state.samples)) * pa.Percentile)
    return state.samples[idx]
}
```

### Merge Operations

Combining partial aggregates from multiple windows:

```go
// For distributed aggregation
type DistributedSum struct {
    local *SumAggregator  // Partial result from this worker
}

func (ds *DistributedSum) Merge(partial1, partial2 interface{}) interface{} {
    // Combine two partial sums
    sum1 := partial1.(float64)
    sum2 := partial2.(float64)
    return sum1 + sum2
}

// Example use case:
// Worker 1 sums events 1-100: 5000
// Worker 2 sums events 101-200: 4800
// Merge result: 9800
```

## Partitioning and State Management

### Event Partitioning

**Goal**: Distribute processing load across workers

**Partitioning Strategy**:
```
Event Stream (all events)
    │
    ├─ Hash by key % 3
    │
    ├─ Partition 0 (mod 0): user_id 1,4,7,10,...
    ├─ Partition 1 (mod 1): user_id 2,5,8,11,...
    └─ Partition 2 (mod 2): user_id 3,6,9,12,...

Each partition processed independently on separate worker
```

**Example Code**:
```go
func getPartition(key string, numPartitions int) int {
    hash := fnv.New32a()
    hash.Write([]byte(key))
    return int(hash.Sum32()) % numPartitions
}

// Route event to correct worker
partition := getPartition(event.Key, numWorkers)
router.Send(event, workers[partition])
```

### Windowed State

**State Storage**:
```go
type WindowState struct {
    window      Window          // Time range
    events      []*Event        // Individual events
    aggregate   interface{}     // Aggregated result
    emitted     bool            // Already sent?
}

// Memory usage per window: ~1KB + 200B per event
```

**Example: 100k events in 1-minute window**:
```
Memory = 1KB (metadata) + (100k × 200B) = ~20MB per window

With 10 concurrent windows = 200MB total
```

### Memory Optimization

**Techniques**:

1. **Discard individual events after aggregation**:
```go
// Instead of keeping all events:
state.events = []*Event{}  // Clear after aggregation
// Save ~200B per event
```

2. **Use pools for event objects**:
```go
var eventPool = sync.Pool{
    New: func() interface{} {
        return &Event{}
    },
}

// Reuse events instead of allocating new
event := eventPool.Get().(*Event)
// ... use event ...
eventPool.Put(event)
```

3. **Compact aggregates**:
```go
// Instead of:
type Aggregate struct {
    sum   float64  // 8 bytes
    count int      // 8 bytes
}

// Use:
type Aggregate struct {
    sum   float32  // 4 bytes (less precision)
    count int32    // 4 bytes
}
```

## CEP Patterns

### Pattern Structure

**Definition**:
```go
type Pattern struct {
    Name       string         // "Multi-City Fraud"
    Conditions []Condition    // What to match
    TimeWindow time.Duration  // How long to track (1 hour)
    Actions    []Action       // What to do (alert)
}
```

**Conditions**:
```go
type Condition struct {
    EventType string                      // "purchase"
    Predicate func(*Event) bool           // amount > 100
    MinCount  int                         // At least N
    MaxCount  int                         // At most N
}
```

### Pattern Matching Algorithm

**Partial Match Tracking**:
```
New event arrives: user_id=123, amount=250, city="NY"

Step 1: Check if matches condition 1
  → event.EventType == "purchase" ✓
  → amount > 100 ✓
  → Create partial match: [{event1}]

Step 2: Check existing partial matches
  → For each partial, check if event matches next condition
  → Accumulate events in partial

Step 3: Check if pattern complete
  → MinCount conditions met? ✓
  → Trigger actions (alert, log, etc.)
```

### Fraud Detection Example

**Pattern**: Detect 5+ purchases > $100 in different cities within 1 hour

**Implementation**:
```go
fraudPattern := &cep.Pattern{
    Name: "Multi-City Fraud",
    Conditions: []cep.Condition{
        {
            EventType: "purchase",
            Predicate: func(e *stream.Event) bool {
                amount := e.Value["amount"].(float64)
                return amount > 100
            },
            MinCount: 5,
        },
    },
    TimeWindow: 1 * time.Hour,
    Actions: []cep.Action{
        &cep.AlertAction{
            AlertType: "fraud",
            Severity:  "high",
        },
    },
}

matcher := cep.NewPatternMatcher(fraudPattern)

for event := range events {
    matcher.Process(event)
    // If pattern detected, alert sent automatically
}
```

### Pattern State Cleanup

**Preventing memory leaks**:
```go
func (pm *PatternMatcher) cleanupExpired() {
    now := time.Now()

    for key, partial := range pm.partialMatches {
        elapsed := now.Sub(partial.startTime)

        if elapsed > pm.pattern.TimeWindow {
            // Pattern window expired
            delete(pm.partialMatches, key)
        }
    }
}
```

**Memory Consideration**:
```
Per partial match: ~500 bytes (events + metadata)
With 1M users: ~500MB worst case
With cleanup: Only active patterns in window
```

## Performance Optimization

### Throughput Optimization

**1. Batch Processing**:
```go
// Instead of:
for event := range events {
    operator.Process(event)  // ~1M events/sec
}

// Do:
batch := make([]*stream.Event, 0, 1000)
for event := range events {
    batch = append(batch, event)
    if len(batch) >= 1000 {
        results := operator.ProcessBatch(batch)
        batch = batch[:0]
    }
}

// Improvement: ~10M events/sec (10x)
```

**2. Vectorized Operations**:
```go
// Process multiple events at once
func (ao *Aggregator) AddBatch(current interface{}, events []*Event) interface{} {
    state := current
    for _, event := range events {
        state = ao.Add(state, event)
    }
    return state
}
```

**3. Parallel Processing**:
```go
// Process different partitions concurrently
workers := 10
for i := 0; i < workers; i++ {
    go func(workerID int) {
        for event := range partitions[workerID] {
            operator.Process(event)
        }
    }(i)
}
```

### Latency Optimization

**1. Window Size Tuning**:
```
Smaller window (10s)  → Lower latency, higher overhead
Larger window (60s)   → Higher latency, lower overhead

Trade-off based on requirements
```

**2. Predicate Optimization**:
```go
// Bad: Multiple type assertions
if amount, ok := e.Value["amount"].(float64); ok {
    if amount > 100 {
        // ... (slow)
    }
}

// Good: Pre-validated
type TypedEvent struct {
    Amount float64
}

if e.Amount > 100 {
    // ... (fast)
}
```

**3. Index-Based Lookups**:
```go
// Instead of:
for _, window := range windows {
    if window.key == lookupKey {
        // Found (O(n))
    }
}

// Use:
windowMap := map[string]*Window{}
foundWindow := windowMap[lookupKey]  // O(1)
```

### Memory Optimization

**1. Object Pooling**:
```go
var statePool = sync.Pool{
    New: func() interface{} {
        return &WindowState{
            events: make([]*Event, 0, 1000),
        }
    },
}

// Use
state := statePool.Get().(*WindowState)
// ... work ...
statePool.Put(state)  // Reuse
```

**2. Compact Data Structures**:
```go
// Before: 64 bytes
type State struct {
    count     int64      // 8 bytes
    sum       float64    // 8 bytes
    min       float64    // 8 bytes
    max       float64    // 8 bytes
    variance  float64    // 8 bytes
    median    float64    // 8 bytes
    //... padding and alignment
}

// After: 40 bytes (40% savings)
type State struct {
    count     int32
    _         int32      // padding
    sum       float32
    min       float32
    max       float32
}
```

## Exactly-Once Semantics

### Problem

```
Worker processes event → Emits result → Crashes
                                          ▲
                                    Result lost!

OR

Worker processes event → Crashes → Replayed
                                    ▲
                            Duplicate result!
```

### Solution: Idempotent Processing

**Using event IDs**:
```go
type Event struct {
    ID        string    // Unique identifier
    Timestamp time.Time
    // ... other fields
}

processedEvents := map[string]bool{}

func processEvent(event *Event) {
    if processedEvents[event.ID] {
        return  // Already processed
    }

    // Process
    result := aggregate(event)

    // Store result with ID
    storeResult(event.ID, result)
    processedEvents[event.ID] = true
}
```

### Checkpointing

**Periodic state snapshots**:
```go
type Checkpoint struct {
    WindowStates  map[string]*WindowState
    Watermark     time.Time
    PartialMatches map[string]*PartialMatch
}

func (wo *WindowOperator) Checkpoint() *Checkpoint {
    return &Checkpoint{
        WindowStates:   deepCopy(wo.windows),
        Watermark:      wo.watermark,
        PartialMatches: deepCopy(wo.patterns),
    }
}

// Save every 5 seconds
ticker := time.NewTicker(5 * time.Second)
for range ticker.C {
    checkpoint := operator.Checkpoint()
    storage.SaveCheckpoint(checkpoint)
}
```

### Recovery from Failure

```
Failure detected
    ↓
Restore latest checkpoint
    ↓
Replay events from checkpoint time
    ↓
Resume normal processing

Result: Exactly-once semantics preserved
```

---

**Advanced Stream Processing Ready!**

For more patterns and examples, see the [examples/queries.md](examples/queries.md) file.
