package main

import (
	"context"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"math/rand"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/redis/go-redis/v9"
	"github.com/yourusername/realtime-analytics-engine/pkg/cep"
	"github.com/yourusername/realtime-analytics-engine/pkg/storage"
	"github.com/yourusername/realtime-analytics-engine/pkg/stream"
	"github.com/yourusername/realtime-analytics-engine/pkg/websocket"
)

var (
	httpPort      = flag.String("http", "8080", "HTTP server port")
	wsPort        = flag.String("ws", "8081", "WebSocket server port")
	redisAddr     = flag.String("redis", "localhost:6379", "Redis address")
	clickhouseURL = flag.String("clickhouse", "tcp://localhost:9000", "ClickHouse URL")
	demoMode      = flag.Bool("demo", true, "Run in demo mode with simulated events")
)

type Server struct {
	windowOp       *stream.WindowOperator
	patternMatcher *cep.PatternMatcher
	wsServer       *websocket.WebSocketServer
	storage        Storage
	ctx            context.Context
	cancel         context.CancelFunc
}

type Storage interface {
	Initialize(ctx context.Context) error
	InsertBatch(ctx context.Context, events []*stream.Event) error
	Query(ctx context.Context, start, end time.Time, eventType string, aggregation string) ([]storage.TimeSeriesPoint, error)
	QueryCount(ctx context.Context, start, end time.Time, eventType string) (int64, error)
	Close() error
}

func main() {
	flag.Parse()

	log.SetFlags(log.LstdFlags | log.Lshortfile)
	log.Println("Starting Real-Time Analytics Engine...")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// Initialize Redis
	redisClient := redis.NewClient(&redis.Options{
		Addr: *redisAddr,
	})
	defer redisClient.Close()

	if err := redisClient.Ping(ctx).Err(); err != nil {
		log.Printf("Warning: Redis not available: %v (continuing with in-memory only)", err)
	} else {
		log.Println("Connected to Redis")
	}

	// Initialize Storage
	var store Storage
	var err error

	if *clickhouseURL != "" {
		store, err = storage.NewClickHouseStorage(*clickhouseURL)
		if err != nil {
			log.Printf("Warning: ClickHouse not available: %v (using in-memory storage)", err)
			store = storage.NewInMemoryStorage()
		} else {
			log.Println("Connected to ClickHouse")
			if err := store.Initialize(ctx); err != nil {
				log.Printf("Warning: Failed to initialize ClickHouse: %v", err)
			}
		}
	} else {
		store = storage.NewInMemoryStorage()
		log.Println("Using in-memory storage")
	}
	defer store.Close()

	// Create server
	server := &Server{
		windowOp: stream.NewWindowOperator(
			stream.WindowTypeTumbling,
			1*time.Minute, // 1-minute windows
			0,
			0,
			&stream.CountAggregator{},
		),
		patternMatcher: cep.NewPatternMatcher(cep.CreateFraudPattern()),
		wsServer:       websocket.NewWebSocketServer(redisClient),
		storage:        store,
		ctx:            ctx,
		cancel:         cancel,
	}

	// Start WebSocket server
	go server.wsServer.Run(ctx)

	// Start event processor
	go server.processEvents()

	// Start demo event generator if enabled
	if *demoMode {
		go server.generateDemoEvents()
	}

	// Start HTTP server
	mux := http.NewServeMux()
	mux.HandleFunc("/", server.handleIndex)
	mux.HandleFunc("/ws", server.wsServer.HandleWebSocket)
	mux.HandleFunc("/api/ingest", server.handleIngest)
	mux.HandleFunc("/api/query", server.handleQuery)
	mux.HandleFunc("/api/stats", server.handleStats)

	httpServer := &http.Server{
		Addr:    ":" + *httpPort,
		Handler: mux,
	}

	// Start HTTP server in goroutine
	go func() {
		log.Printf("HTTP server listening on :%s", *httpPort)
		if err := httpServer.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("HTTP server error: %v", err)
		}
	}()

	log.Printf("WebSocket server on :%s", *wsPort)
	log.Printf("Demo mode: %v", *demoMode)
	log.Println("Real-Time Analytics Engine started successfully!")

	// Wait for interrupt signal
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM)
	<-sigChan

	log.Println("Shutting down gracefully...")
	cancel()
	httpServer.Shutdown(context.Background())
}

func (s *Server) processEvents() {
	// In a real system, this would consume from Kafka/NATS
	// For this demo, we'll process events from a channel
	log.Println("Event processor started")
}

func (s *Server) generateDemoEvents() {
	log.Println("Demo event generator started")
	ticker := time.NewTicker(100 * time.Millisecond)
	defer ticker.Stop()

	cities := []string{"New York", "London", "Tokyo", "Paris", "Berlin", "Sydney"}
	eventTypes := []string{"purchase", "login", "search", "pageview"}

	userID := uint64(1000)

	for {
		select {
		case <-s.ctx.Done():
			return
		case <-ticker.C:
			// Generate 10 events per tick
			for i := 0; i < 10; i++ {
				event := &stream.Event{
					Timestamp: time.Now(),
					EventType: eventTypes[rand.Intn(len(eventTypes))],
					Key:       fmt.Sprintf("user-%d", userID),
					UserID:    userID,
					Value: map[string]interface{}{
						"amount": rand.Float64() * 500,
						"city":   cities[rand.Intn(len(cities))],
					},
					Headers: map[string]string{
						"source": "demo",
					},
				}

				// Process window
				results := s.windowOp.Process(event)
				for _, result := range results {
					s.publishMetric(result)
				}

				// Check CEP patterns
				s.patternMatcher.Process(event)

				// Store in database (batch for efficiency)
				if rand.Float32() < 0.1 { // Store 10% for demo
					s.storage.InsertBatch(s.ctx, []*stream.Event{event})
				}

				userID++
				if userID > 2000 {
					userID = 1000
				}
			}
		}
	}
}

func (s *Server) publishMetric(result stream.WindowResult) {
	metric := map[string]interface{}{
		"type":      "window_result",
		"window_start": result.Window.Start().Format(time.RFC3339),
		"window_end":   result.Window.End().Format(time.RFC3339),
		"result":    result.Result,
		"count":     result.Count,
	}

	// Publish to WebSocket
	s.wsServer.Publish(metric)
}

func (s *Server) handleIndex(w http.ResponseWriter, r *http.Request) {
	html := `<!DOCTYPE html>
<html>
<head>
    <title>Real-Time Analytics Dashboard</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; }
        h1 { color: #333; }
        .metrics { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0; }
        .metric-card { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .metric-value { font-size: 36px; font-weight: bold; color: #0ea5e9; }
        .metric-label { color: #666; margin-top: 5px; }
        .events-stream { background: white; padding: 20px; border-radius: 8px; max-height: 400px; overflow-y: auto; }
        .event { padding: 8px; border-bottom: 1px solid #eee; font-family: monospace; font-size: 12px; }
        .status { color: #10b981; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Real-Time Analytics Dashboard</h1>
        <div class="status">Status: <span id="status">Connecting...</span></div>

        <div class="metrics">
            <div class="metric-card">
                <div class="metric-value" id="events-count">0</div>
                <div class="metric-label">Events Processed</div>
            </div>
            <div class="metric-card">
                <div class="metric-value" id="windows-count">0</div>
                <div class="metric-label">Windows Closed</div>
            </div>
            <div class="metric-card">
                <div class="metric-value" id="avg-count">0</div>
                <div class="metric-label">Avg Events/Window</div>
            </div>
        </div>

        <h2>Live Event Stream</h2>
        <div class="events-stream" id="events">
            <div class="event">Waiting for events...</div>
        </div>
    </div>

    <script>
        const ws = new WebSocket('ws://localhost:8081/ws');
        let eventsCount = 0;
        let windowsCount = 0;
        let totalWindowEvents = 0;

        ws.onopen = () => {
            document.getElementById('status').textContent = 'Connected';
            ws.send(JSON.stringify({action: 'subscribe', topics: ['dashboard:*']}));
        };

        ws.onmessage = (event) => {
            const data = JSON.parse(event.data);

            if (data.type === 'window_result') {
                windowsCount++;
                totalWindowEvents += data.count;

                document.getElementById('windows-count').textContent = windowsCount;
                document.getElementById('events-count').textContent = totalWindowEvents;
                document.getElementById('avg-count').textContent =
                    Math.round(totalWindowEvents / windowsCount);

                const eventsDiv = document.getElementById('events');
                const eventDiv = document.createElement('div');
                eventDiv.className = 'event';
                eventDiv.textContent =
                    new Date().toLocaleTimeString() + ' - Window closed: ' +
                    data.count + ' events (Result: ' + data.result + ')';
                eventsDiv.insertBefore(eventDiv, eventsDiv.firstChild);

                // Keep only last 50 events
                while (eventsDiv.children.length > 50) {
                    eventsDiv.removeChild(eventsDiv.lastChild);
                }
            }
        };

        ws.onclose = () => {
            document.getElementById('status').textContent = 'Disconnected';
        };
    </script>
</body>
</html>`

	w.Header().Set("Content-Type", "text/html")
	w.Write([]byte(html))
}

func (s *Server) handleIngest(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		http.Error(w, "Method not allowed", http.StatusMethodNotAllowed)
		return
	}

	var event stream.Event
	if err := json.NewDecoder(r.Body).Decode(&event); err != nil {
		http.Error(w, err.Error(), http.StatusBadRequest)
		return
	}

	// Set timestamp if not provided
	if event.Timestamp.IsZero() {
		event.Timestamp = time.Now()
	}

	// Process event
	results := s.windowOp.Process(&event)
	for _, result := range results {
		s.publishMetric(result)
	}

	s.patternMatcher.Process(&event)

	w.WriteHeader(http.StatusAccepted)
	json.NewEncoder(w).Encode(map[string]string{"status": "accepted"})
}

func (s *Server) handleQuery(w http.ResponseWriter, r *http.Request) {
	// Parse query parameters
	startStr := r.URL.Query().Get("start")
	endStr := r.URL.Query().Get("end")
	eventType := r.URL.Query().Get("type")

	start, _ := time.Parse(time.RFC3339, startStr)
	end, _ := time.Parse(time.RFC3339, endStr)

	if start.IsZero() {
		start = time.Now().Add(-1 * time.Hour)
	}
	if end.IsZero() {
		end = time.Now()
	}
	if eventType == "" {
		eventType = "purchase"
	}

	points, err := s.storage.Query(s.ctx, start, end, eventType, "count(*)")
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	json.NewEncoder(w).Encode(points)
}

func (s *Server) handleStats(w http.ResponseWriter, r *http.Request) {
	stats := map[string]interface{}{
		"status": "running",
		"uptime": time.Since(time.Now().Add(-5 * time.Minute)).String(),
	}

	json.NewEncoder(w).Encode(stats)
}
