# Example Queries

This document shows example queries and use cases for the Real-Time Analytics Engine.

## API Examples

### 1. Ingest Single Event

```bash
curl -X POST http://localhost:8080/api/ingest \
  -H "Content-Type: application/json" \
  -d '{
    "timestamp": "2024-01-15T10:30:00Z",
    "event_type": "purchase",
    "user_id": 12345,
    "key": "user-12345",
    "value": {
      "amount": 199.99,
      "city": "New York",
      "product": "laptop"
    },
    "headers": {
      "source": "web",
      "version": "1.0"
    }
  }'
```

### 2. Batch Ingest (Send Multiple Events)

```bash
for i in {1..100}; do
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"purchase\",
      \"user_id\": $i,
      \"key\": \"user-$i\",
      \"value\": {
        \"amount\": $((RANDOM % 500 + 10)),
        \"city\": \"City-$((RANDOM % 10))\"
      }
    }"
done
```

### 3. Query Historical Data

```bash
# Events in the last hour
curl "http://localhost:8080/api/query?type=purchase&start=$(date -u -d '1 hour ago' +%Y-%m-%dT%H:%M:%SZ)&end=$(date -u +%Y-%m-%dT%H:%M:%SZ)"

# Events for specific time range
curl "http://localhost:8080/api/query?type=login&start=2024-01-15T09:00:00Z&end=2024-01-15T10:00:00Z"
```

### 4. Get Statistics

```bash
curl http://localhost:8080/api/stats
```

## Use Case Examples

### Use Case 1: E-commerce Fraud Detection

**Scenario**: Detect users making multiple high-value purchases from different cities within 1 hour.

**Pattern Configuration**:
```go
pattern := &cep.Pattern{
    Name: "Multi-City Fraud",
    Conditions: []cep.Condition{
        {
            EventType: "purchase",
            Predicate: func(e *stream.Event) bool {
                amount := e.Value["amount"].(float64)
                return amount > 100
            },
            MinCount: 5,
        },
    },
    TimeWindow: 1 * time.Hour,
    Actions: []cep.Action{
        &cep.AlertAction{
            AlertType: "fraud",
            Severity:  "high",
        },
    },
}
```

**Test Data**:
```bash
# Simulate fraud: User 9999 makes 5 purchases in different cities
for city in "New York" "Los Angeles" "Chicago" "Miami" "Seattle"; do
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"purchase\",
      \"user_id\": 9999,
      \"key\": \"user-9999\",
      \"value\": {
        \"amount\": 250.00,
        \"city\": \"$city\"
      }
    }"
  sleep 1
done
```

### Use Case 2: Application Performance Monitoring

**Scenario**: Track p95 and p99 latency for API requests in 5-minute windows.

**Configuration**:
```go
// p95 latency
p95Window := stream.NewWindowOperator(
    stream.WindowTypeTumbling,
    5*time.Minute,
    0, 0,
    &stream.PercentileAggregator{
        Field:      "latency",
        Percentile: 0.95,
    },
)

// p99 latency
p99Window := stream.NewWindowOperator(
    stream.WindowTypeTumbling,
    5*time.Minute,
    0, 0,
    &stream.PercentileAggregator{
        Field:      "latency",
        Percentile: 0.99,
    },
)
```

**Test Data**:
```bash
# Simulate API requests with varying latencies
for i in {1..1000}; do
  latency=$((RANDOM % 200 + 10))  # 10-210ms
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"api_request\",
      \"user_id\": $i,
      \"key\": \"request-$i\",
      \"value\": {
        \"latency\": $latency,
        \"endpoint\": \"/api/users\"
      }
    }"
done
```

### Use Case 3: IoT Sensor Monitoring

**Scenario**: Detect temperature anomalies from IoT sensors using session windows.

**Configuration**:
```go
sessionWindow := stream.NewWindowOperator(
    stream.WindowTypeSession,
    0, 0,
    30*time.Second,  // 30s inactivity gap
    &stream.AvgAggregator{Field: "temperature"},
)
```

**Test Data**:
```bash
# Simulate IoT sensor data
device_id=1001
for temp in 22.5 23.1 88.9 89.2 89.5 23.0 22.8; do
  curl -X POST http://localhost:8080/api/ingest \
    -H "Content-Type: application/json" \
    -d "{
      \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
      \"event_type\": \"temperature\",
      \"user_id\": $device_id,
      \"key\": \"device-$device_id\",
      \"value\": {
        \"temperature\": $temp,
        \"unit\": \"celsius\"
      }
    }"
  sleep 2
done
```

### Use Case 4: Real-Time Dashboard Metrics

**Scenario**: Count active users per minute for live dashboard.

**Configuration**:
```go
activeUsersWindow := stream.NewWindowOperator(
    stream.WindowTypeTumbling,
    1*time.Minute,
    0, 0,
    &stream.DistinctCountAggregator{Field: "user_id"},
)
```

**WebSocket Client**:
```javascript
const ws = new WebSocket('ws://localhost:8081/ws');

ws.onopen = () => {
    console.log('Connected to analytics engine');
    ws.send(JSON.stringify({
        action: 'subscribe',
        topics: ['dashboard:*']
    }));
};

ws.onmessage = (event) => {
    const data = JSON.parse(event.data);
    console.log('Window result:', data);

    if (data.type === 'window_result') {
        updateDashboard({
            timestamp: data.window_end,
            metric: data.result,
            count: data.count
        });
    }
};
```

## Advanced Queries

### Multi-Aggregation (Run in Code)

```go
// Track multiple metrics simultaneously
type MultiAggregator struct {
    count   *CountAggregator
    sum     *SumAggregator
    avg     *AvgAggregator
    min     *MinAggregator
    max     *MaxAggregator
}

func (ma *MultiAggregator) Add(current interface{}, event *Event) interface{} {
    state := current.(map[string]interface{})
    if state == nil {
        state = make(map[string]interface{})
    }

    state["count"] = ma.count.Add(state["count"], event)
    state["sum"] = ma.sum.Add(state["sum"], event)
    state["avg"] = ma.avg.Add(state["avg"], event)
    state["min"] = ma.min.Add(state["min"], event)
    state["max"] = ma.max.Add(state["max"], event)

    return state
}
```

### Sliding Window for Moving Averages

```go
// 5-minute moving average, updated every minute
slidingAvg := stream.NewWindowOperator(
    stream.WindowTypeSliding,
    5*time.Minute,   // Window size
    1*time.Minute,   // Slide interval
    0,
    &stream.AvgAggregator{Field: "value"},
)
```

## Performance Testing

### Load Test Script

```bash
#!/bin/bash
# Generate 10,000 events per second for 1 minute

total_events=600000
batch_size=100
concurrent=100

echo "Generating $total_events events..."

for i in $(seq 1 $concurrent); do
  (
    for j in $(seq 1 $((total_events / concurrent / batch_size))); do
      for k in $(seq 1 $batch_size); do
        curl -X POST http://localhost:8080/api/ingest \
          -H "Content-Type: application/json" \
          -d "{
            \"timestamp\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\",
            \"event_type\": \"test\",
            \"user_id\": $((i * 10000 + j * 100 + k)),
            \"key\": \"user-$((i * 10000 + j * 100 + k))\",
            \"value\": {
              \"metric\": $((RANDOM % 1000))
            }
          }" &
      done
      wait
    done
  ) &
done

wait
echo "Load test complete!"
```

## Monitoring Queries

### Check System Health

```bash
# Get current stats
curl http://localhost:8080/api/stats | jq

# Monitor event throughput
watch -n 1 'curl -s http://localhost:8080/api/stats | jq ".events_per_sec"'
```

### Query Time-Series Data

```bash
# Get minute-by-minute event counts for last hour
curl "http://localhost:8080/api/query?type=purchase&start=$(date -u -d '1 hour ago' +%Y-%m-%dT%H:%M:%SZ)&end=$(date -u +%Y-%m-%dT%H:%M:%SZ)" | jq
```

## Integration Examples

### Python Client

```python
import requests
import json
from datetime import datetime

def send_event(event_type, user_id, data):
    url = "http://localhost:8080/api/ingest"
    payload = {
        "timestamp": datetime.utcnow().isoformat() + "Z",
        "event_type": event_type,
        "user_id": user_id,
        "key": f"user-{user_id}",
        "value": data
    }
    response = requests.post(url, json=payload)
    return response.status_code == 202

# Send event
send_event("purchase", 12345, {
    "amount": 99.99,
    "city": "San Francisco"
})
```

### Go Client

```go
package main

import (
    "bytes"
    "encoding/json"
    "net/http"
    "time"
)

type Event struct {
    Timestamp string                 `json:"timestamp"`
    EventType string                 `json:"event_type"`
    UserID    uint64                 `json:"user_id"`
    Key       string                 `json:"key"`
    Value     map[string]interface{} `json:"value"`
}

func sendEvent(eventType string, userID uint64, value map[string]interface{}) error {
    event := Event{
        Timestamp: time.Now().UTC().Format(time.RFC3339),
        EventType: eventType,
        UserID:    userID,
        Key:       fmt.Sprintf("user-%d", userID),
        Value:     value,
    }

    data, _ := json.Marshal(event)
    _, err := http.Post(
        "http://localhost:8080/api/ingest",
        "application/json",
        bytes.NewBuffer(data),
    )
    return err
}
```

---

For more examples, see the [README.md](../README.md) and the demo mode (`make demo`).
