package cep

import (
	"fmt"
	"log"
	"sync"
	"time"

	"github.com/yourusername/realtime-analytics-engine/pkg/stream"
)

// Pattern defines a sequence of events to match
type Pattern struct {
	Name       string
	Conditions []Condition
	TimeWindow time.Duration
	Actions    []Action
}

type Condition struct {
	EventType string
	Predicate func(event *stream.Event) bool
	MinCount  int // Minimum occurrences
	MaxCount  int // Maximum occurrences (0 = unlimited)
}

type Action interface {
	Execute(matches []*stream.Event)
}

// PatternMatcher detects patterns in event stream
type PatternMatcher struct {
	pattern        *Pattern
	partialMatches map[string]*PartialMatch
	mu             sync.RWMutex
}

type PartialMatch struct {
	events         []*stream.Event
	startTime      time.Time
	conditionIndex int
}

func NewPatternMatcher(pattern *Pattern) *PatternMatcher {
	return &PatternMatcher{
		pattern:        pattern,
		partialMatches: make(map[string]*PartialMatch),
	}
}

// Process event and check for pattern matches
func (pm *PatternMatcher) Process(event *stream.Event) {
	pm.mu.Lock()
	defer pm.mu.Unlock()

	// Clean up expired partial matches
	pm.cleanupExpired()

	// Check if event matches first condition (start new partial match)
	firstCondition := pm.pattern.Conditions[0]
	if event.EventType == firstCondition.EventType &&
		firstCondition.Predicate(event) {

		key := fmt.Sprintf("%s-%d", event.Key, time.Now().UnixNano())
		pm.partialMatches[key] = &PartialMatch{
			events:         []*stream.Event{event},
			startTime:      event.Timestamp,
			conditionIndex: 0,
		}
	}

	// Check existing partial matches
	for key, partial := range pm.partialMatches {
		// Skip if window expired
		if time.Since(partial.startTime) > pm.pattern.TimeWindow {
			delete(pm.partialMatches, key)
			continue
		}

		// Check next condition
		nextConditionIndex := partial.conditionIndex + 1
		if nextConditionIndex >= len(pm.pattern.Conditions) {
			continue // Pattern already complete
		}

		nextCondition := pm.pattern.Conditions[nextConditionIndex]

		// Check if event matches next condition
		if event.EventType == nextCondition.EventType &&
			nextCondition.Predicate(event) {

			// Add event to partial match
			partial.events = append(partial.events, event)

			// Count events for this condition
			count := 0
			for _, e := range partial.events {
				if e.EventType == nextCondition.EventType {
					count++
				}
			}

			// Check if condition satisfied
			if count >= nextCondition.MinCount {
				partial.conditionIndex = nextConditionIndex

				// Check if all conditions satisfied
				if partial.conditionIndex == len(pm.pattern.Conditions)-1 {
					// Pattern matched! Execute actions
					for _, action := range pm.pattern.Actions {
						action.Execute(partial.events)
					}

					// Remove completed match
					delete(pm.partialMatches, key)
				}
			}
		}
	}
}

func (pm *PatternMatcher) cleanupExpired() {
	now := time.Now()
	for key, partial := range pm.partialMatches {
		if now.Sub(partial.startTime) > pm.pattern.TimeWindow {
			delete(pm.partialMatches, key)
		}
	}
}

// Alert represents a detected pattern match
type Alert struct {
	Type      string
	Severity  string
	Message   string
	Events    []*stream.Event
	Timestamp time.Time
}

// AlertAction sends alert when pattern matches
type AlertAction struct {
	AlertType string
	Severity  string
}

func (aa *AlertAction) Execute(matches []*stream.Event) {
	// Extract unique cities
	cities := make(map[string]bool)
	totalAmount := 0.0

	for _, event := range matches {
		if city, ok := event.Value["city"].(string); ok {
			cities[city] = true
		}
		if amount, ok := event.Value["amount"].(float64); ok {
			totalAmount += amount
		}
	}

	// Alert if suspicious pattern detected
	if len(cities) >= 3 {
		alert := Alert{
			Type:      aa.AlertType,
			Severity:  aa.Severity,
			Message:   fmt.Sprintf("User %d made %d purchases in %d cities (total: $%.2f)", matches[0].UserID, len(matches), len(cities), totalAmount),
			Events:    matches,
			Timestamp: time.Now(),
		}
		sendAlert(alert)
	}
}

func sendAlert(alert Alert) {
	// Send to webhook, Slack, email, etc.
	log.Printf("🚨 ALERT [%s/%s]: %s", alert.Type, alert.Severity, alert.Message)
}

// CreateFraudPattern creates a fraud detection pattern
func CreateFraudPattern() *Pattern {
	return &Pattern{
		Name: "Multi-City Fraud",
		Conditions: []Condition{
			{
				EventType: "purchase",
				Predicate: func(e *stream.Event) bool {
					amount, ok := e.Value["amount"].(float64)
					return ok && amount > 100
				},
				MinCount: 5,
			},
		},
		TimeWindow: 1 * time.Hour,
		Actions: []Action{
			&AlertAction{
				AlertType: "fraud",
				Severity:  "high",
			},
		},
	}
}

// LogAction logs pattern matches
type LogAction struct {
	Message string
}

func (la *LogAction) Execute(matches []*stream.Event) {
	log.Printf("Pattern matched: %s (events: %d)", la.Message, len(matches))
}

// WebhookAction calls a webhook on pattern match
type WebhookAction struct {
	URL string
}

func (wa *WebhookAction) Execute(matches []*stream.Event) {
	// In a real implementation, this would make an HTTP POST to the webhook URL
	log.Printf("Webhook action: POST %s (events: %d)", wa.URL, len(matches))
}
