package storage

import (
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"time"

	_ "github.com/ClickHouse/clickhouse-go/v2"
	"github.com/yourusername/realtime-analytics-engine/pkg/stream"
)

type ClickHouseStorage struct {
	db *sql.DB
}

func NewClickHouseStorage(dsn string) (*ClickHouseStorage, error) {
	db, err := sql.Open("clickhouse", dsn)
	if err != nil {
		return nil, err
	}

	// Test connection
	if err := db.Ping(); err != nil {
		return nil, fmt.Errorf("failed to ping ClickHouse: %w", err)
	}

	return &ClickHouseStorage{db: db}, nil
}

// Initialize schema
func (chs *ClickHouseStorage) Initialize(ctx context.Context) error {
	query := `
	CREATE TABLE IF NOT EXISTS events (
		timestamp DateTime,
		event_type String,
		user_id UInt64,
		key String,
		value String,
		metadata String
	) ENGINE = MergeTree()
	PARTITION BY toYYYYMM(timestamp)
	ORDER BY (user_id, timestamp)
	SETTINGS index_granularity = 8192
	`

	_, err := chs.db.ExecContext(ctx, query)
	return err
}

// Insert events in batch
func (chs *ClickHouseStorage) InsertBatch(ctx context.Context, events []*stream.Event) error {
	if len(events) == 0 {
		return nil
	}

	tx, err := chs.db.BeginTx(ctx, nil)
	if err != nil {
		return err
	}
	defer tx.Rollback()

	stmt, err := tx.PrepareContext(ctx, `
		INSERT INTO events (timestamp, event_type, user_id, key, value, metadata)
		VALUES (?, ?, ?, ?, ?, ?)
	`)
	if err != nil {
		return err
	}
	defer stmt.Close()

	for _, event := range events {
		valueJSON, _ := json.Marshal(event.Value)
		metadataJSON, _ := json.Marshal(event.Headers)

		_, err := stmt.ExecContext(ctx,
			event.Timestamp,
			event.EventType,
			event.UserID,
			event.Key,
			string(valueJSON),
			string(metadataJSON),
		)
		if err != nil {
			return err
		}
	}

	return tx.Commit()
}

// Query with time range and aggregations
func (chs *ClickHouseStorage) Query(
	ctx context.Context,
	start, end time.Time,
	eventType string,
	aggregation string,
) ([]TimeSeriesPoint, error) {

	query := fmt.Sprintf(`
		SELECT
			toStartOfMinute(timestamp) as time,
			%s as value
		FROM events
		WHERE timestamp BETWEEN ? AND ?
		AND event_type = ?
		GROUP BY time
		ORDER BY time
	`, aggregation)

	rows, err := chs.db.QueryContext(ctx, query, start, end, eventType)
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var points []TimeSeriesPoint
	for rows.Next() {
		var point TimeSeriesPoint
		if err := rows.Scan(&point.Time, &point.Value); err != nil {
			return nil, err
		}
		points = append(points, point)
	}

	return points, rows.Err()
}

// QueryCount returns count of events
func (chs *ClickHouseStorage) QueryCount(
	ctx context.Context,
	start, end time.Time,
	eventType string,
) (int64, error) {
	query := `
		SELECT count(*) as count
		FROM events
		WHERE timestamp BETWEEN ? AND ?
		AND event_type = ?
	`

	var count int64
	err := chs.db.QueryRowContext(ctx, query, start, end, eventType).Scan(&count)
	return count, err
}

// Close closes the database connection
func (chs *ClickHouseStorage) Close() error {
	return chs.db.Close()
}

type TimeSeriesPoint struct {
	Time  time.Time
	Value float64
}

// InMemoryStorage provides in-memory storage for testing/development
type InMemoryStorage struct {
	events []*stream.Event
}

func NewInMemoryStorage() *InMemoryStorage {
	return &InMemoryStorage{
		events: make([]*stream.Event, 0),
	}
}

func (ims *InMemoryStorage) Initialize(ctx context.Context) error {
	return nil
}

func (ims *InMemoryStorage) InsertBatch(ctx context.Context, events []*stream.Event) error {
	ims.events = append(ims.events, events...)
	return nil
}

func (ims *InMemoryStorage) Query(
	ctx context.Context,
	start, end time.Time,
	eventType string,
	aggregation string,
) ([]TimeSeriesPoint, error) {
	// Simple in-memory aggregation
	points := make(map[time.Time]float64)

	for _, event := range ims.events {
		if event.EventType == eventType &&
			!event.Timestamp.Before(start) &&
			!event.Timestamp.After(end) {

			minute := event.Timestamp.Truncate(time.Minute)

			// Simple COUNT aggregation
			if aggregation == "count(*)" {
				points[minute]++
			}
		}
	}

	// Convert to slice
	result := make([]TimeSeriesPoint, 0, len(points))
	for t, v := range points {
		result = append(result, TimeSeriesPoint{Time: t, Value: v})
	}

	return result, nil
}

func (ims *InMemoryStorage) QueryCount(
	ctx context.Context,
	start, end time.Time,
	eventType string,
) (int64, error) {
	var count int64
	for _, event := range ims.events {
		if event.EventType == eventType &&
			!event.Timestamp.Before(start) &&
			!event.Timestamp.After(end) {
			count++
		}
	}
	return count, nil
}

func (ims *InMemoryStorage) Close() error {
	return nil
}
