package stream

import (
	"math"
	"sort"
)

// CountAggregator counts events in window
type CountAggregator struct{}

func (ca *CountAggregator) Add(current interface{}, event *Event) interface{} {
	if current == nil {
		return 1
	}
	return current.(int) + 1
}

func (ca *CountAggregator) Merge(a, b interface{}) interface{} {
	return a.(int) + b.(int)
}

func (ca *CountAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0
	}
	return aggregate
}

// SumAggregator sums numeric field
type SumAggregator struct {
	Field string
}

func (sa *SumAggregator) Add(current interface{}, event *Event) interface{} {
	var value float64
	if v, ok := event.Value[sa.Field].(float64); ok {
		value = v
	} else if v, ok := event.Value[sa.Field].(int); ok {
		value = float64(v)
	}

	if current == nil {
		return value
	}
	return current.(float64) + value
}

func (sa *SumAggregator) Merge(a, b interface{}) interface{} {
	return a.(float64) + b.(float64)
}

func (sa *SumAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0.0
	}
	return aggregate
}

// AvgAggregator calculates average
type AvgAggregator struct {
	Field string
}

type avgState struct {
	sum   float64
	count int
}

func (aa *AvgAggregator) Add(current interface{}, event *Event) interface{} {
	var value float64
	if v, ok := event.Value[aa.Field].(float64); ok {
		value = v
	} else if v, ok := event.Value[aa.Field].(int); ok {
		value = float64(v)
	}

	if current == nil {
		return &avgState{sum: value, count: 1}
	}

	state := current.(*avgState)
	return &avgState{
		sum:   state.sum + value,
		count: state.count + 1,
	}
}

func (aa *AvgAggregator) Merge(a, b interface{}) interface{} {
	stateA := a.(*avgState)
	stateB := b.(*avgState)
	return &avgState{
		sum:   stateA.sum + stateB.sum,
		count: stateA.count + stateB.count,
	}
}

func (aa *AvgAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0.0
	}
	state := aggregate.(*avgState)
	if state.count == 0 {
		return 0.0
	}
	return state.sum / float64(state.count)
}

// PercentileAggregator calculates percentiles (p50, p95, p99)
type PercentileAggregator struct {
	Field      string
	Percentile float64 // 0.5 for p50, 0.95 for p95, etc.
}

func (pa *PercentileAggregator) Add(current interface{}, event *Event) interface{} {
	var value float64
	if v, ok := event.Value[pa.Field].(float64); ok {
		value = v
	} else if v, ok := event.Value[pa.Field].(int); ok {
		value = float64(v)
	}

	if current == nil {
		return []float64{value}
	}

	values := current.([]float64)
	return append(values, value)
}

func (pa *PercentileAggregator) Merge(a, b interface{}) interface{} {
	valuesA := a.([]float64)
	valuesB := b.([]float64)
	return append(valuesA, valuesB...)
}

func (pa *PercentileAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0.0
	}

	values := aggregate.([]float64)
	if len(values) == 0 {
		return 0.0
	}

	// Sort values
	sort.Float64s(values)

	// Calculate percentile index
	index := int(math.Ceil(float64(len(values)) * pa.Percentile)) - 1
	if index < 0 {
		index = 0
	}
	if index >= len(values) {
		index = len(values) - 1
	}

	return values[index]
}

// DistinctCountAggregator counts distinct values
type DistinctCountAggregator struct {
	Field string
}

func (dca *DistinctCountAggregator) Add(current interface{}, event *Event) interface{} {
	value := event.Value[dca.Field].(string)

	if current == nil {
		return map[string]bool{value: true}
	}

	distinct := current.(map[string]bool)
	distinct[value] = true
	return distinct
}

func (dca *DistinctCountAggregator) Merge(a, b interface{}) interface{} {
	distinctA := a.(map[string]bool)
	distinctB := b.(map[string]bool)

	merged := make(map[string]bool)
	for k := range distinctA {
		merged[k] = true
	}
	for k := range distinctB {
		merged[k] = true
	}
	return merged
}

func (dca *DistinctCountAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0
	}
	return len(aggregate.(map[string]bool))
}

// MinAggregator finds minimum value
type MinAggregator struct {
	Field string
}

func (ma *MinAggregator) Add(current interface{}, event *Event) interface{} {
	var value float64
	if v, ok := event.Value[ma.Field].(float64); ok {
		value = v
	} else if v, ok := event.Value[ma.Field].(int); ok {
		value = float64(v)
	}

	if current == nil {
		return value
	}

	currentMin := current.(float64)
	if value < currentMin {
		return value
	}
	return currentMin
}

func (ma *MinAggregator) Merge(a, b interface{}) interface{} {
	valA := a.(float64)
	valB := b.(float64)
	if valA < valB {
		return valA
	}
	return valB
}

func (ma *MinAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0.0
	}
	return aggregate
}

// MaxAggregator finds maximum value
type MaxAggregator struct {
	Field string
}

func (ma *MaxAggregator) Add(current interface{}, event *Event) interface{} {
	var value float64
	if v, ok := event.Value[ma.Field].(float64); ok {
		value = v
	} else if v, ok := event.Value[ma.Field].(int); ok {
		value = float64(v)
	}

	if current == nil {
		return value
	}

	currentMax := current.(float64)
	if value > currentMax {
		return value
	}
	return currentMax
}

func (ma *MaxAggregator) Merge(a, b interface{}) interface{} {
	valA := a.(float64)
	valB := b.(float64)
	if valA > valB {
		return valA
	}
	return valB
}

func (ma *MaxAggregator) Result(aggregate interface{}) interface{} {
	if aggregate == nil {
		return 0.0
	}
	return aggregate
}
