package stream

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestCountAggregator(t *testing.T) {
	agg := &CountAggregator{}

	var result interface{}
	for i := 0; i < 10; i++ {
		event := &Event{}
		result = agg.Add(result, event)
	}

	assert.Equal(t, 10, agg.Result(result))
}

func TestSumAggregator(t *testing.T) {
	agg := &SumAggregator{Field: "amount"}

	var result interface{}
	for i := 1; i <= 5; i++ {
		event := &Event{
			Value: map[string]interface{}{
				"amount": float64(i * 10),
			},
		}
		result = agg.Add(result, event)
	}

	// Sum of 10, 20, 30, 40, 50 = 150
	assert.Equal(t, 150.0, agg.Result(result))
}

func TestAvgAggregator(t *testing.T) {
	agg := &AvgAggregator{Field: "latency"}

	var result interface{}
	for i := 1; i <= 5; i++ {
		event := &Event{
			Value: map[string]interface{}{
				"latency": float64(i * 10),
			},
		}
		result = agg.Add(result, event)
	}

	// Average of 10, 20, 30, 40, 50 = 30
	assert.Equal(t, 30.0, agg.Result(result))
}

func TestPercentileAggregator(t *testing.T) {
	agg := &PercentileAggregator{
		Field:      "latency",
		Percentile: 0.95,
	}

	// Add 100 events with latencies 1-100ms
	var result interface{}
	for i := 1; i <= 100; i++ {
		event := &Event{
			Value: map[string]interface{}{
				"latency": float64(i),
			},
		}
		result = agg.Add(result, event)
	}

	// p95 should be 95
	assert.Equal(t, 95.0, agg.Result(result))
}

func TestDistinctCountAggregator(t *testing.T) {
	agg := &DistinctCountAggregator{Field: "city"}

	var result interface{}
	cities := []string{"NYC", "London", "Tokyo", "NYC", "London", "Paris"}
	for _, city := range cities {
		event := &Event{
			Value: map[string]interface{}{
				"city": city,
			},
		}
		result = agg.Add(result, event)
	}

	// Should have 4 distinct cities
	assert.Equal(t, 4, agg.Result(result))
}

func TestMinAggregator(t *testing.T) {
	agg := &MinAggregator{Field: "temperature"}

	var result interface{}
	temps := []float64{25.5, 23.1, 28.9, 22.0, 26.5}
	for _, temp := range temps {
		event := &Event{
			Value: map[string]interface{}{
				"temperature": temp,
			},
		}
		result = agg.Add(result, event)
	}

	assert.Equal(t, 22.0, agg.Result(result))
}

func TestMaxAggregator(t *testing.T) {
	agg := &MaxAggregator{Field: "temperature"}

	var result interface{}
	temps := []float64{25.5, 23.1, 28.9, 22.0, 26.5}
	for _, temp := range temps {
		event := &Event{
			Value: map[string]interface{}{
				"temperature": temp,
			},
		}
		result = agg.Add(result, event)
	}

	assert.Equal(t, 28.9, agg.Result(result))
}

func BenchmarkCountAggregator(b *testing.B) {
	agg := &CountAggregator{}
	event := &Event{}

	var result interface{}
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		result = agg.Add(result, event)
	}
}

func BenchmarkPercentileAggregator(b *testing.B) {
	agg := &PercentileAggregator{
		Field:      "latency",
		Percentile: 0.95,
	}
	event := &Event{
		Value: map[string]interface{}{
			"latency": 42.0,
		},
	}

	var result interface{}
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		result = agg.Add(result, event)
	}
}
