package stream

import (
	"time"
)

// Event represents a single data point in the stream
type Event struct {
	Timestamp time.Time
	EventType string
	Key       string
	UserID    uint64
	Value     map[string]interface{}
	Headers   map[string]string
}

// Window represents a time-based grouping of events
type Window interface {
	Contains(event *Event) bool
	Start() time.Time
	End() time.Time
	IsExpired(watermark time.Time) bool
}

// TumblingWindow groups events into non-overlapping fixed-size windows
type TumblingWindow struct {
	start    time.Time
	duration time.Duration
}

func NewTumblingWindow(start time.Time, duration time.Duration) *TumblingWindow {
	return &TumblingWindow{
		start:    start.Truncate(duration),
		duration: duration,
	}
}

func (tw *TumblingWindow) Contains(event *Event) bool {
	return !event.Timestamp.Before(tw.start) &&
		event.Timestamp.Before(tw.start.Add(tw.duration))
}

func (tw *TumblingWindow) Start() time.Time {
	return tw.start
}

func (tw *TumblingWindow) End() time.Time {
	return tw.start.Add(tw.duration)
}

func (tw *TumblingWindow) IsExpired(watermark time.Time) bool {
	return watermark.After(tw.End()) || watermark.Equal(tw.End())
}

// SlidingWindow groups events into overlapping windows
type SlidingWindow struct {
	start time.Time
	size  time.Duration
	slide time.Duration
}

func NewSlidingWindow(start time.Time, size, slide time.Duration) *SlidingWindow {
	return &SlidingWindow{
		start: start,
		size:  size,
		slide: slide,
	}
}

func (sw *SlidingWindow) Contains(event *Event) bool {
	return !event.Timestamp.Before(sw.start) &&
		event.Timestamp.Before(sw.start.Add(sw.size))
}

func (sw *SlidingWindow) Start() time.Time {
	return sw.start
}

func (sw *SlidingWindow) End() time.Time {
	return sw.start.Add(sw.size)
}

func (sw *SlidingWindow) IsExpired(watermark time.Time) bool {
	return watermark.After(sw.start.Add(sw.size)) || watermark.Equal(sw.start.Add(sw.size))
}

// SessionWindow groups events by inactivity gap
type SessionWindow struct {
	start       time.Time
	lastEvent   time.Time
	gapDuration time.Duration
}

func NewSessionWindow(start time.Time, gap time.Duration) *SessionWindow {
	return &SessionWindow{
		start:       start,
		lastEvent:   start,
		gapDuration: gap,
	}
}

func (sw *SessionWindow) Contains(event *Event) bool {
	// Event belongs to session if within gap from last event
	return event.Timestamp.Sub(sw.lastEvent) <= sw.gapDuration
}

func (sw *SessionWindow) Start() time.Time {
	return sw.start
}

func (sw *SessionWindow) End() time.Time {
	return sw.lastEvent.Add(sw.gapDuration)
}

func (sw *SessionWindow) AddEvent(event *Event) {
	if event.Timestamp.After(sw.lastEvent) {
		sw.lastEvent = event.Timestamp
	}
}

func (sw *SessionWindow) IsExpired(watermark time.Time) bool {
	return watermark.Sub(sw.lastEvent) > sw.gapDuration
}
