package stream

import (
	"fmt"
	"sync"
	"time"
)

// WindowOperator manages windows and aggregates events
type WindowOperator struct {
	windowType  WindowType
	windowSize  time.Duration
	windowSlide time.Duration // For sliding windows
	sessionGap  time.Duration // For session windows
	aggregator  Aggregator
	watermark   time.Time
	windows     map[string]*WindowState
	mu          sync.RWMutex
}

type WindowType int

const (
	WindowTypeTumbling WindowType = iota
	WindowTypeSliding
	WindowTypeSession
)

type WindowState struct {
	window    Window
	events    []*Event
	aggregate interface{}
	emitted   bool
}

type Aggregator interface {
	Add(current interface{}, event *Event) interface{}
	Merge(a, b interface{}) interface{}
	Result(aggregate interface{}) interface{}
}

func NewWindowOperator(windowType WindowType, size, slide, gap time.Duration, agg Aggregator) *WindowOperator {
	return &WindowOperator{
		windowType:  windowType,
		windowSize:  size,
		windowSlide: slide,
		sessionGap:  gap,
		aggregator:  agg,
		watermark:   time.Time{},
		windows:     make(map[string]*WindowState),
	}
}

// Process incoming event
func (wo *WindowOperator) Process(event *Event) []WindowResult {
	wo.mu.Lock()
	defer wo.mu.Unlock()

	results := []WindowResult{}

	// Update watermark (max event timestamp - allowed lateness)
	if event.Timestamp.After(wo.watermark) {
		newWatermark := event.Timestamp.Add(-5 * time.Second) // 5s lateness
		if newWatermark.After(wo.watermark) {
			wo.watermark = newWatermark
			// Close and emit expired windows
			results = append(results, wo.closeExpiredWindows()...)
		}
	}

	// Assign event to windows
	windows := wo.assignWindows(event)

	for _, window := range windows {
		windowKey := wo.windowKey(window)

		// Get or create window state
		state, exists := wo.windows[windowKey]
		if !exists {
			state = &WindowState{
				window:    window,
				events:    []*Event{},
				aggregate: nil,
			}
			wo.windows[windowKey] = state
		}

		// Add event to window
		state.events = append(state.events, event)

		// Update aggregate
		state.aggregate = wo.aggregator.Add(state.aggregate, event)
	}

	return results
}

// Assign event to appropriate windows based on window type
func (wo *WindowOperator) assignWindows(event *Event) []Window {
	switch wo.windowType {
	case WindowTypeTumbling:
		window := NewTumblingWindow(event.Timestamp, wo.windowSize)
		return []Window{window}

	case WindowTypeSliding:
		// Sliding window belongs to multiple overlapping windows
		windows := []Window{}
		windowStart := event.Timestamp.Truncate(wo.windowSlide)

		// Find all windows this event belongs to
		for i := 0; i < int(wo.windowSize/wo.windowSlide); i++ {
			start := windowStart.Add(-time.Duration(i) * wo.windowSlide)
			window := NewSlidingWindow(start, wo.windowSize, wo.windowSlide)
			if window.Contains(event) {
				windows = append(windows, window)
			}
		}
		return windows

	case WindowTypeSession:
		// Find existing session or create new one
		for _, state := range wo.windows {
			if sessionWindow, ok := state.window.(*SessionWindow); ok {
				if sessionWindow.Contains(event) {
					sessionWindow.AddEvent(event)
					return []Window{sessionWindow}
				}
			}
		}
		// No existing session, create new one
		return []Window{NewSessionWindow(event.Timestamp, wo.sessionGap)}
	}

	return []Window{}
}

// Close and emit expired windows
func (wo *WindowOperator) closeExpiredWindows() []WindowResult {
	results := []WindowResult{}

	for key, state := range wo.windows {
		if state.window.IsExpired(wo.watermark) && !state.emitted {
			// Emit window result
			result := wo.aggregator.Result(state.aggregate)
			results = append(results, WindowResult{
				Window: state.window,
				Result: result,
				Count:  len(state.events),
			})

			state.emitted = true

			// Clean up window
			delete(wo.windows, key)
		}
	}

	return results
}

func (wo *WindowOperator) windowKey(window Window) string {
	return fmt.Sprintf("%d-%d", window.Start().Unix(), window.End().Unix())
}

type WindowResult struct {
	Window Window
	Result interface{}
	Count  int
}
