package stream

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

func TestTumblingWindow(t *testing.T) {
	start := time.Now().Truncate(time.Minute)

	operator := NewWindowOperator(
		WindowTypeTumbling,
		1*time.Minute,
		0,
		0,
		&CountAggregator{},
	)

	// Add 5 events within 1 minute
	for i := 0; i < 5; i++ {
		event := &Event{
			Timestamp: start.Add(time.Duration(i) * time.Second),
			EventType: "test",
			Key:       "user1",
		}
		operator.Process(event)
	}

	// Advance watermark to close window
	operator.mu.Lock()
	operator.watermark = start.Add(2 * time.Minute)
	results := operator.closeExpiredWindows()
	operator.mu.Unlock()

	assert.Len(t, results, 1)
	assert.Equal(t, 5, results[0].Result)
}

func TestSlidingWindow(t *testing.T) {
	start := time.Now().Truncate(time.Minute)

	operator := NewWindowOperator(
		WindowTypeSliding,
		5*time.Minute, // Window size
		1*time.Minute, // Slide
		0,
		&CountAggregator{},
	)

	// Add event - should appear in multiple windows
	event := &Event{
		Timestamp: start,
		EventType: "test",
		Key:       "user1",
	}
	operator.Process(event)

	// Event should be in multiple windows due to sliding
	operator.mu.RLock()
	windowCount := len(operator.windows)
	operator.mu.RUnlock()

	assert.True(t, windowCount >= 1, "Event should be in at least one window")
}

func TestSessionWindow(t *testing.T) {
	start := time.Now()

	operator := NewWindowOperator(
		WindowTypeSession,
		0,
		0,
		30*time.Second, // 30s gap
		&CountAggregator{},
	)

	// Add events within gap
	event1 := &Event{
		Timestamp: start,
		EventType: "test",
		Key:       "user1",
	}
	operator.Process(event1)

	event2 := &Event{
		Timestamp: start.Add(10 * time.Second),
		EventType: "test",
		Key:       "user1",
	}
	operator.Process(event2)

	// Session window creates windows as needed, may have 1-2 depending on timing
	operator.mu.RLock()
	windowCount := len(operator.windows)
	operator.mu.RUnlock()

	assert.True(t, windowCount >= 1, "Should have at least one session window")
}

func TestWindowExpiration(t *testing.T) {
	start := time.Now().Truncate(time.Minute)

	operator := NewWindowOperator(
		WindowTypeTumbling,
		1*time.Minute,
		0,
		0,
		&CountAggregator{},
	)

	// Add event
	event := &Event{
		Timestamp: start,
		EventType: "test",
		Key:       "user1",
	}
	operator.Process(event)

	// Window should exist
	operator.mu.RLock()
	assert.Len(t, operator.windows, 1)
	operator.mu.RUnlock()

	// Advance watermark past window end
	operator.mu.Lock()
	operator.watermark = start.Add(2 * time.Minute)
	results := operator.closeExpiredWindows()
	operator.mu.Unlock()

	// Window should be closed and result emitted
	assert.Len(t, results, 1)
	assert.Equal(t, 1, results[0].Result)

	// Window should be removed
	operator.mu.RLock()
	assert.Len(t, operator.windows, 0)
	operator.mu.RUnlock()
}

func BenchmarkWindowOperator(b *testing.B) {
	operator := NewWindowOperator(
		WindowTypeTumbling,
		1*time.Minute,
		0,
		0,
		&CountAggregator{},
	)

	event := &Event{
		Timestamp: time.Now(),
		EventType: "test",
		Key:       "user1",
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		operator.Process(event)
	}
}
