# Go IDE Platform - API Reference

Complete API reference for WebSocket messages and command protocols.

## Table of Contents

1. [Message Protocol](#message-protocol)
2. [LSP API](#lsp-api)
3. [DAP API](#dap-api)
4. [Git API](#git-api)
5. [File API](#file-api)
6. [Terminal API](#terminal-api)
7. [Plugin API](#plugin-api)
8. [Type Definitions](#type-definitions)
9. [Error Codes](#error-codes)

---

## Message Protocol

### Base Message Format

```typescript
interface Message {
  id: string;           // Unique request ID for correlation
  type: string;         // Message type: lsp|dap|git|file|terminal|plugin
  command: string;      // Command name within type
  params?: object;      // Command parameters
}
```

### Response Format

```typescript
interface Response {
  id: string;           // Correlates to request ID
  type: string;         // Same as request type
  result?: any;         // Response data (if successful)
  error?: string;       // Error message (if failed)
}
```

### Error Response

```typescript
{
  "id": "req-123",
  "type": "lsp",
  "error": "gopls not found in PATH"
}
```

---

## LSP API

Language Server Protocol integration via gopls.

### Commands

#### completion

Get code completions at cursor position.

**Request:**
```json
{
  "id": "req-1",
  "type": "lsp",
  "command": "completion",
  "params": {
    "uri": "file:///workspace/main.go",
    "line": 10,
    "column": 5
  }
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "isIncomplete": false,
    "items": [
      {
        "label": "fmt.Println",
        "kind": 6,
        "detail": "func Println(...interface{}) (int, error)",
        "documentation": "Println formats using the default formats...",
        "sortText": "0",
        "filterText": "Println"
      }
    ]
  }
}
```

**Parameters:**
- `uri` (string): File URI (e.g., "file:///workspace/main.go")
- `line` (number): 0-based line number
- `column` (number): 0-based column number

**Returns:**
- `isIncomplete` (boolean): More items available
- `items` (CompletionItem[]): List of completion suggestions

---

#### hover

Get type information and documentation on hover.

**Request:**
```json
{
  "id": "req-2",
  "type": "lsp",
  "command": "hover",
  "params": {
    "uri": "file:///workspace/main.go",
    "line": 10,
    "column": 5
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "contents": "func Println(a ...interface{}) (n int, err error)",
    "range": {
      "start": {"line": 10, "column": 4},
      "end": {"line": 10, "column": 11}
    }
  }
}
```

**Parameters:**
- `uri` (string): File URI
- `line` (number): Line number
- `column` (number): Column number

**Returns:**
- `contents` (string): Hover text (formatted as markdown)
- `range` (Range): The range of the hovered symbol

---

#### definition

Jump to symbol definition.

**Request:**
```json
{
  "id": "req-3",
  "type": "lsp",
  "command": "definition",
  "params": {
    "uri": "file:///workspace/main.go",
    "line": 10,
    "column": 5
  }
}
```

**Response:**
```json
{
  "id": "req-3",
  "result": {
    "uri": "file:///go/src/fmt/print.go",
    "range": {
      "start": {"line": 254, "column": 5},
      "end": {"line": 254, "column": 12}
    }
  }
}
```

**Parameters:**
- `uri` (string): File URI
- `line` (number): Line number
- `column` (number): Column number

**Returns:**
- `uri` (string): URI of definition file
- `range` (Range): Location in definition file

---

#### references

Find all references to a symbol.

**Request:**
```json
{
  "id": "req-4",
  "type": "lsp",
  "command": "references",
  "params": {
    "uri": "file:///workspace/main.go",
    "line": 10,
    "column": 5,
    "includeDeclaration": true
  }
}
```

**Response:**
```json
{
  "id": "req-4",
  "result": {
    "locations": [
      {
        "uri": "file:///workspace/main.go",
        "range": {"start": {"line": 10, "column": 4}, "end": {"line": 10, "column": 11}}
      },
      {
        "uri": "file:///workspace/utils.go",
        "range": {"start": {"line": 45, "column": 8}, "end": {"line": 45, "column": 15}}
      }
    ]
  }
}
```

**Parameters:**
- `uri` (string): File URI
- `line` (number): Line number
- `column` (number): Column number
- `includeDeclaration` (boolean, optional): Include declaration in results

**Returns:**
- `locations` (Location[]): Array of reference locations

---

#### format

Format document code.

**Request:**
```json
{
  "id": "req-5",
  "type": "lsp",
  "command": "format",
  "params": {
    "uri": "file:///workspace/main.go"
  }
}
```

**Response:**
```json
{
  "id": "req-5",
  "result": {
    "edits": [
      {
        "range": {"start": {"line": 5, "column": 0}, "end": {"line": 5, "column": 10}},
        "newText": "func main() {"
      }
    ]
  }
}
```

**Parameters:**
- `uri` (string): File URI
- `options` (FormattingOptions, optional): Formatting options

**Returns:**
- `edits` (TextEdit[]): Formatting edits to apply

---

#### codeActions

Get available code actions.

**Request:**
```json
{
  "id": "req-6",
  "type": "lsp",
  "command": "codeActions",
  "params": {
    "uri": "file:///workspace/main.go",
    "range": {
      "start": {"line": 10, "column": 0},
      "end": {"line": 10, "column": 20}
    }
  }
}
```

**Response:**
```json
{
  "id": "req-6",
  "result": {
    "actions": [
      {
        "title": "Organize imports",
        "kind": "source.organizeImports",
        "edit": {
          "changes": {
            "file:///workspace/main.go": [...]
          }
        }
      }
    ]
  }
}
```

**Parameters:**
- `uri` (string): File URI
- `range` (Range): Code range

**Returns:**
- `actions` (CodeAction[]): Available code actions

---

## DAP API

Debug Adapter Protocol integration via Delve.

### Commands

#### createSession

Create a new debug session.

**Request:**
```json
{
  "id": "req-1",
  "type": "dap",
  "command": "createSession",
  "params": {
    "sessionId": "debug-1"
  }
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "sessionId": "debug-1",
    "status": "created"
  }
}
```

**Parameters:**
- `sessionId` (string): Unique session identifier

**Returns:**
- `sessionId` (string): Created session ID
- `status` (string): Session status

---

#### launch

Launch a debug session.

**Request:**
```json
{
  "id": "req-2",
  "type": "dap",
  "command": "launch",
  "params": {
    "sessionId": "debug-1",
    "mode": "debug",
    "program": "/workspace/bin/myapp",
    "args": ["--flag=value"],
    "cwd": "/workspace",
    "env": {"DEBUG": "1"}
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "sessionId": "debug-1",
    "pid": 12345,
    "status": "running"
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `mode` (string): "debug" or "test"
- `program` (string): Program path
- `args` (string[], optional): Program arguments
- `cwd` (string, optional): Working directory
- `env` (object, optional): Environment variables

**Returns:**
- `sessionId` (string): Session ID
- `pid` (number): Process ID
- `status` (string): Session status

---

#### attach

Attach to running process.

**Request:**
```json
{
  "id": "req-3",
  "type": "dap",
  "command": "attach",
  "params": {
    "sessionId": "debug-1",
    "pid": 12345
  }
}
```

**Response:**
```json
{
  "id": "req-3",
  "result": {
    "sessionId": "debug-1",
    "status": "attached"
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `pid` (number): Process ID to attach to

**Returns:**
- `sessionId` (string): Session ID
- `status` (string): Session status

---

#### setBreakpoint

Set a breakpoint.

**Request:**
```json
{
  "id": "req-4",
  "type": "dap",
  "command": "setBreakpoint",
  "params": {
    "sessionId": "debug-1",
    "file": "/workspace/main.go",
    "line": 42,
    "condition": "i > 5",
    "hitCondition": "5"
  }
}
```

**Response:**
```json
{
  "id": "req-4",
  "result": {
    "breakpointId": "bp-1",
    "file": "/workspace/main.go",
    "line": 42,
    "verified": true
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `file` (string): File path
- `line` (number): Line number
- `condition` (string, optional): Break condition
- `hitCondition` (string, optional): Hit count condition

**Returns:**
- `breakpointId` (string): Breakpoint identifier
- `file` (string): File path
- `line` (number): Actual line (may differ)
- `verified` (boolean): Whether breakpoint was set

---

#### removeBreakpoint

Remove a breakpoint.

**Request:**
```json
{
  "id": "req-5",
  "type": "dap",
  "command": "removeBreakpoint",
  "params": {
    "sessionId": "debug-1",
    "breakpointId": "bp-1"
  }
}
```

**Response:**
```json
{
  "id": "req-5",
  "result": {
    "success": true
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `breakpointId` (string): Breakpoint identifier

**Returns:**
- `success` (boolean): Removal success

---

#### continue

Continue execution.

**Request:**
```json
{
  "id": "req-6",
  "type": "dap",
  "command": "continue",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1
  }
}
```

**Response:**
```json
{
  "id": "req-6",
  "result": {
    "allThreadsContinued": false
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `threadId` (number): Thread identifier

**Returns:**
- `allThreadsContinued` (boolean): All threads continued

---

#### stepIn

Step into function.

**Request:**
```json
{
  "id": "req-7",
  "type": "dap",
  "command": "stepIn",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1
  }
}
```

**Response:**
```json
{
  "id": "req-7",
  "result": {
    "threadId": 1,
    "reason": "step"
  }
}
```

---

#### stepOver

Step over line.

**Request:**
```json
{
  "id": "req-8",
  "type": "dap",
  "command": "stepOver",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1
  }
}
```

---

#### stepOut

Step out of function.

**Request:**
```json
{
  "id": "req-9",
  "type": "dap",
  "command": "stepOut",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1
  }
}
```

---

#### pause

Pause execution.

**Request:**
```json
{
  "id": "req-10",
  "type": "dap",
  "command": "pause",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1
  }
}
```

---

#### stackTrace

Get call stack.

**Request:**
```json
{
  "id": "req-11",
  "type": "dap",
  "command": "stackTrace",
  "params": {
    "sessionId": "debug-1",
    "threadId": 1,
    "levels": 20
  }
}
```

**Response:**
```json
{
  "id": "req-11",
  "result": {
    "stackFrames": [
      {
        "id": 0,
        "name": "main",
        "source": {"path": "/workspace/main.go"},
        "line": 42,
        "column": 5
      },
      {
        "id": 1,
        "name": "func1",
        "source": {"path": "/workspace/utils.go"},
        "line": 10,
        "column": 0
      }
    ]
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `threadId` (number): Thread identifier
- `levels` (number, optional): Max frames to return

**Returns:**
- `stackFrames` (StackFrame[]): Call stack frames

---

#### variables

Get variables in scope.

**Request:**
```json
{
  "id": "req-12",
  "type": "dap",
  "command": "variables",
  "params": {
    "sessionId": "debug-1",
    "frameId": 0
  }
}
```

**Response:**
```json
{
  "id": "req-12",
  "result": {
    "variables": [
      {
        "name": "x",
        "value": "42",
        "type": "int"
      },
      {
        "name": "name",
        "value": "\"John\"",
        "type": "string"
      }
    ]
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `frameId` (number): Frame identifier

**Returns:**
- `variables` (Variable[]): Variables in scope

---

#### evaluate

Evaluate expression.

**Request:**
```json
{
  "id": "req-13",
  "type": "dap",
  "command": "evaluate",
  "params": {
    "sessionId": "debug-1",
    "frameId": 0,
    "expression": "x + 5"
  }
}
```

**Response:**
```json
{
  "id": "req-13",
  "result": {
    "result": "47",
    "type": "int"
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `frameId` (number): Frame identifier
- `expression` (string): Expression to evaluate

**Returns:**
- `result` (string): Expression result
- `type` (string): Result type

---

## Git API

Git operations via go-git.

### Commands

#### status

Get repository status.

**Request:**
```json
{
  "id": "req-1",
  "type": "git",
  "command": "status",
  "params": {}
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "branch": "main",
    "modified": ["main.go", "README.md"],
    "staged": ["go.mod"],
    "untracked": [".env", "bin/"],
    "ahead": 3,
    "behind": 0,
    "isClean": false
  }
}
```

**Returns:**
- `branch` (string): Current branch name
- `modified` (string[]): Modified files
- `staged` (string[]): Staged files
- `untracked` (string[]): Untracked files
- `ahead` (number): Commits ahead of remote
- `behind` (number): Commits behind remote
- `isClean` (boolean): No uncommitted changes

---

#### diff

Get diff.

**Request:**
```json
{
  "id": "req-2",
  "type": "git",
  "command": "diff",
  "params": {
    "file": "main.go"
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "diff": "--- a/main.go\n+++ b/main.go\n@@ -1,5 +1,6 @@\n...",
    "stats": {
      "additions": 10,
      "deletions": 2
    }
  }
}
```

**Parameters:**
- `file` (string, optional): Specific file (all if omitted)

**Returns:**
- `diff` (string): Unified diff format
- `stats` (DiffStats): Diff statistics

---

#### commit

Create commit.

**Request:**
```json
{
  "id": "req-3",
  "type": "git",
  "command": "commit",
  "params": {
    "message": "feat: add new feature",
    "files": ["main.go", "README.md"],
    "author": {
      "name": "John Doe",
      "email": "john@example.com"
    }
  }
}
```

**Response:**
```json
{
  "id": "req-3",
  "result": {
    "hash": "abc123def456",
    "author": "John Doe <john@example.com>",
    "message": "feat: add new feature",
    "timestamp": 1699564800000
  }
}
```

**Parameters:**
- `message` (string): Commit message
- `files` (string[]): Files to commit
- `author` (Author): Author information

**Returns:**
- `hash` (string): Commit hash
- `author` (string): Author info
- `message` (string): Commit message
- `timestamp` (number): Commit timestamp

---

#### branch

Branch operations.

**List branches:**
```json
{
  "id": "req-4",
  "type": "git",
  "command": "branch",
  "params": {
    "operation": "list"
  }
}
```

**Response:**
```json
{
  "id": "req-4",
  "result": {
    "branches": [
      {"name": "main", "isHead": true},
      {"name": "develop", "isHead": false}
    ]
  }
}
```

**Checkout branch:**
```json
{
  "id": "req-5",
  "type": "git",
  "command": "branch",
  "params": {
    "operation": "checkout",
    "branch": "develop"
  }
}
```

**Create branch:**
```json
{
  "id": "req-6",
  "type": "git",
  "command": "branch",
  "params": {
    "operation": "create",
    "branch": "feature/new-feature"
  }
}
```

**Delete branch:**
```json
{
  "id": "req-7",
  "type": "git",
  "command": "branch",
  "params": {
    "operation": "delete",
    "branch": "old-branch"
  }
}
```

---

#### log

Get commit history.

**Request:**
```json
{
  "id": "req-8",
  "type": "git",
  "command": "log",
  "params": {
    "limit": 20,
    "file": "main.go"
  }
}
```

**Response:**
```json
{
  "id": "req-8",
  "result": {
    "commits": [
      {
        "hash": "abc123",
        "author": "John Doe",
        "message": "feat: add feature",
        "timestamp": 1699564800000
      }
    ]
  }
}
```

**Parameters:**
- `limit` (number, optional): Max commits to return
- `file` (string, optional): Only commits for file

**Returns:**
- `commits` (Commit[]): Commit history

---

## File API

File system operations.

### Commands

#### read

Read file contents.

**Request:**
```json
{
  "id": "req-1",
  "type": "file",
  "command": "read",
  "params": {
    "path": "main.go"
  }
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "path": "main.go",
    "content": "package main\n\nfunc main() {...}",
    "size": 1024,
    "modified": 1699564800000
  }
}
```

**Parameters:**
- `path` (string): File path relative to workspace

**Returns:**
- `path` (string): File path
- `content` (string): File contents
- `size` (number): File size in bytes
- `modified` (number): Last modification time

---

#### write

Write file contents.

**Request:**
```json
{
  "id": "req-2",
  "type": "file",
  "command": "write",
  "params": {
    "path": "main.go",
    "content": "package main\n\nfunc main() {...}"
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "path": "main.go",
    "size": 1024
  }
}
```

**Parameters:**
- `path` (string): File path
- `content` (string): File contents

**Returns:**
- `path` (string): File path
- `size` (number): Written size

---

#### delete

Delete file or directory.

**Request:**
```json
{
  "id": "req-3",
  "type": "file",
  "command": "delete",
  "params": {
    "path": "old_file.go"
  }
}
```

**Response:**
```json
{
  "id": "req-3",
  "result": {
    "success": true,
    "path": "old_file.go"
  }
}
```

---

#### rename

Rename file or directory.

**Request:**
```json
{
  "id": "req-4",
  "type": "file",
  "command": "rename",
  "params": {
    "from": "old_name.go",
    "to": "new_name.go"
  }
}
```

---

#### mkdir

Create directory.

**Request:**
```json
{
  "id": "req-5",
  "type": "file",
  "command": "mkdir",
  "params": {
    "path": "pkg/utils"
  }
}
```

---

#### search

Search files.

**Request:**
```json
{
  "id": "req-6",
  "type": "file",
  "command": "search",
  "params": {
    "query": "TODO",
    "regex": false,
    "includePattern": "*.go",
    "excludePattern": "vendor/**"
  }
}
```

**Response:**
```json
{
  "id": "req-6",
  "result": {
    "results": [
      {
        "file": "main.go",
        "line": 42,
        "column": 10,
        "text": "// TODO: implement feature"
      }
    ],
    "total": 1
  }
}
```

---

#### tree

Get file tree.

**Request:**
```json
{
  "id": "req-7",
  "type": "file",
  "command": "tree",
  "params": {
    "path": ".",
    "depth": 3
  }
}
```

**Response:**
```json
{
  "id": "req-7",
  "result": {
    "name": "workspace",
    "type": "directory",
    "children": [
      {
        "name": "main.go",
        "type": "file",
        "size": 1024
      },
      {
        "name": "cmd",
        "type": "directory",
        "children": [...]
      }
    ]
  }
}
```

---

## Terminal API

Terminal operations.

### Commands

#### createSession

Create terminal session.

**Request:**
```json
{
  "id": "req-1",
  "type": "terminal",
  "command": "createSession",
  "params": {
    "sessionId": "term-1"
  }
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "sessionId": "term-1",
    "status": "active"
  }
}
```

---

#### execute

Execute command.

**Request:**
```json
{
  "id": "req-2",
  "type": "terminal",
  "command": "execute",
  "params": {
    "sessionId": "term-1",
    "command": "go test ./...",
    "timeout": 30000
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "stdout": "ok    github.com/yourorg/go-ide    0.123s\n",
    "stderr": "",
    "exitCode": 0,
    "duration": 125
  }
}
```

**Parameters:**
- `sessionId` (string): Session identifier
- `command` (string): Shell command
- `timeout` (number, optional): Timeout in milliseconds

**Returns:**
- `stdout` (string): Standard output
- `stderr` (string): Standard error
- `exitCode` (number): Exit code
- `duration` (number): Execution duration

---

## Plugin API

Plugin management.

### Commands

#### list

List plugins.

**Request:**
```json
{
  "id": "req-1",
  "type": "plugin",
  "command": "list",
  "params": {}
}
```

**Response:**
```json
{
  "id": "req-1",
  "result": {
    "plugins": [
      {
        "id": "plugin-formatter",
        "name": "Go Formatter",
        "version": "1.0.0",
        "active": true,
        "permissions": ["file:read", "file:write"]
      }
    ],
    "total": 1
  }
}
```

---

#### load

Load plugin.

**Request:**
```json
{
  "id": "req-2",
  "type": "plugin",
  "command": "load",
  "params": {
    "name": "Go Formatter",
    "version": "1.0.0",
    "author": "Your Name",
    "permissions": ["file:read", "file:write"]
  }
}
```

**Response:**
```json
{
  "id": "req-2",
  "result": {
    "id": "plugin-formatter",
    "name": "Go Formatter",
    "active": true
  }
}
```

---

#### unload

Unload plugin.

**Request:**
```json
{
  "id": "req-3",
  "type": "plugin",
  "command": "unload",
  "params": {
    "pluginId": "plugin-formatter"
  }
}
```

**Response:**
```json
{
  "id": "req-3",
  "result": {
    "success": true,
    "pluginId": "plugin-formatter"
  }
}
```

---

#### info

Get plugin info.

**Request:**
```json
{
  "id": "req-4",
  "type": "plugin",
  "command": "info",
  "params": {
    "pluginId": "plugin-formatter"
  }
}
```

**Response:**
```json
{
  "id": "req-4",
  "result": {
    "id": "plugin-formatter",
    "name": "Go Formatter",
    "version": "1.0.0",
    "description": "Formats Go code using gofmt",
    "author": "Your Name",
    "permissions": ["file:read", "file:write"],
    "active": true
  }
}
```

---

## Type Definitions

### Common Types

#### Range

```typescript
interface Range {
  start: Position;
  end: Position;
}
```

#### Position

```typescript
interface Position {
  line: number;      // 0-based
  column: number;    // 0-based
}
```

#### Location

```typescript
interface Location {
  uri: string;
  range: Range;
}
```

#### TextEdit

```typescript
interface TextEdit {
  range: Range;
  newText: string;
}
```

#### CompletionItem

```typescript
interface CompletionItem {
  label: string;
  kind: number;
  detail?: string;
  documentation?: string;
  sortText?: string;
  filterText?: string;
}
```

#### Hover

```typescript
interface Hover {
  contents: string;
  range?: Range;
}
```

---

## Error Codes

Standard error codes returned by the API.

| Code | Message | Description |
|------|---------|-------------|
| `NOT_FOUND` | Resource not found | File, symbol, or session doesn't exist |
| `INVALID_PATH` | Invalid path | Path traversal or syntax error |
| `PERMISSION_DENIED` | Permission denied | Access outside workspace |
| `TOOL_NOT_FOUND` | Tool not found | gopls, delve, or git not available |
| `PARSE_ERROR` | Parse error | Failed to parse code or config |
| `TIMEOUT` | Operation timeout | Command exceeded timeout |
| `INVALID_PARAMS` | Invalid parameters | Missing or malformed parameters |
| `INTERNAL_ERROR` | Internal error | Unexpected server error |

---

**API Reference Version**: 1.0.0
**Last Updated**: 2025-10-29
