# Plugin Development Guide

Complete guide for developing plugins for the Go IDE Platform.

## Table of Contents

1. [Overview](#overview)
2. [Plugin Types](#plugin-types)
3. [Go Plugin Development](#go-plugin-development)
4. [WASM Plugin Development](#wasm-plugin-development)
5. [Plugin Lifecycle](#plugin-lifecycle)
6. [Permission System](#permission-system)
7. [Plugin API Reference](#plugin-api-reference)
8. [Examples](#examples)
9. [Testing](#testing)
10. [Distribution](#distribution)

---

## Overview

The Go IDE Platform supports two types of plugins:

1. **Go Plugins**: Native performance, full access to IDE APIs
2. **WASM Plugins**: Sandboxed, language-agnostic, secure

Both plugin types use a common interface and lifecycle model.

### Plugin Capabilities

Plugins can:
- Extend editor functionality
- Add custom commands
- Integrate with external tools
- Provide code analysis and linting
- Format code
- Manage project files
- Access IDE APIs
- Listen for editor events

### Plugin Restrictions

Plugins cannot:
- Escape sandbox (WASM)
- Access resources outside workspace
- Modify IDE core
- Affect other plugins
- Use restricted permissions

---

## Plugin Types

### Go Plugins

**Characteristics:**
- Native Go code compiled as shared library
- Full performance
- Full Go API access
- Shared memory with IDE
- Require trust (can access any system resource)

**When to use:**
- Performance-critical operations
- Complex integrations
- Full IDE API access needed
- Trusted code only

**Building:**
```bash
go build -buildmode=plugin -o my-plugin.so .
```

### WASM Plugins

**Characteristics:**
- WebAssembly bytecode
- Sandboxed execution
- Language-agnostic
- Portable across platforms
- Permission-based access

**When to use:**
- Untrusted third-party plugins
- Cross-platform compatibility
- Lightweight operations
- Plugin marketplace distribution

**Building:**
```bash
GOOS=js GOARCH=wasm go build -o my-plugin.wasm .
```

---

## Go Plugin Development

### Project Structure

```
my-plugin/
├── main.go              # Plugin entry point
├── handler.go           # Command handlers
├── utils.go             # Helper functions
├── go.mod
├── go.sum
├── plugin.json          # Plugin metadata
├── tests/
│   └── handler_test.go  # Tests
└── README.md            # Documentation
```

### Plugin Interface

All plugins must implement this interface:

```go
type Plugin interface {
    // Name returns the plugin identifier
    Name() string

    // Version returns semantic version
    Version() string

    // Description returns human-readable description
    Description() string

    // Activate is called when plugin is loaded
    Activate() error

    // Deactivate is called when plugin is unloaded
    Deactivate() error

    // Execute processes WebSocket commands
    Execute(command string, params map[string]interface{}) (interface{}, error)
}
```

### Basic Plugin Template

**File: main.go**

```go
package main

import (
    "fmt"
    "log"
)

// MyPlugin implements the Plugin interface
type MyPlugin struct {
    name        string
    version     string
    initialized bool
}

// NewPlugin creates a new plugin instance
func NewPlugin() *MyPlugin {
    return &MyPlugin{
        name:    "my-plugin",
        version: "1.0.0",
    }
}

// Name returns plugin name
func (p *MyPlugin) Name() string {
    return p.name
}

// Version returns plugin version
func (p *MyPlugin) Version() string {
    return p.version
}

// Description returns plugin description
func (p *MyPlugin) Description() string {
    return "My custom IDE plugin"
}

// Activate is called when plugin is loaded
func (p *MyPlugin) Activate() error {
    log.Printf("Plugin %s v%s activated\n", p.name, p.version)
    p.initialized = true
    return nil
}

// Deactivate is called when plugin is unloaded
func (p *MyPlugin) Deactivate() error {
    log.Printf("Plugin %s deactivated\n", p.name)
    p.initialized = false
    return nil
}

// Execute processes commands from WebSocket
func (p *MyPlugin) Execute(command string, params map[string]interface{}) (interface{}, error) {
    if !p.initialized {
        return nil, fmt.Errorf("plugin not initialized")
    }

    switch command {
    case "hello":
        return p.handleHello(params)
    case "process":
        return p.handleProcess(params)
    default:
        return nil, fmt.Errorf("unknown command: %s", command)
    }
}

// handleHello handles the hello command
func (p *MyPlugin) handleHello(params map[string]interface{}) (interface{}, error) {
    name, ok := params["name"].(string)
    if !ok {
        return nil, fmt.Errorf("name parameter required")
    }

    return map[string]interface{}{
        "message": fmt.Sprintf("Hello, %s!", name),
    }, nil
}

// handleProcess handles the process command
func (p *MyPlugin) handleProcess(params map[string]interface{}) (interface{}, error) {
    data, ok := params["data"].(string)
    if !ok {
        return nil, fmt.Errorf("data parameter required")
    }

    // Process the data
    result := processData(data)

    return map[string]interface{}{
        "input":  data,
        "output": result,
    }, nil
}

// processData is a helper function
func processData(data string) string {
    // Your processing logic here
    return data
}

// Export function for plugin loading
var Plugin *MyPlugin

func init() {
    Plugin = NewPlugin()
}
```

### Building Go Plugin

**File: go.mod**

```go
module github.com/yourorg/my-plugin

go 1.21

// No external dependencies for basic plugin
```

**Build command:**

```bash
# Build as shared library
go build -buildmode=plugin -o my-plugin.so .

# Verify plugin loads
file my-plugin.so
```

### Plugin Metadata

**File: plugin.json**

```json
{
  "id": "my-plugin",
  "name": "My Plugin",
  "version": "1.0.0",
  "author": "Your Name",
  "description": "Description of what plugin does",
  "type": "go",
  "main": "my-plugin.so",
  "entry": "Plugin",
  "permissions": [
    "file:read",
    "file:write"
  ],
  "keywords": ["formatter", "linter"],
  "homepage": "https://github.com/yourorg/my-plugin",
  "repository": {
    "type": "git",
    "url": "https://github.com/yourorg/my-plugin.git"
  }
}
```

---

## WASM Plugin Development

### Project Structure

```
my-wasm-plugin/
├── main.go              # WASM entry point
├── handler.go           # Handler functions
├── go.mod
├── plugin.json          # Metadata
└── README.md            # Documentation
```

### WASM Plugin Template

**File: main.go**

```go
package main

import (
    "encoding/json"
    "syscall/js"
)

// PluginInfo contains plugin metadata
type PluginInfo struct {
    ID          string   `json:"id"`
    Name        string   `json:"name"`
    Version     string   `json:"version"`
    Description string   `json:"description"`
    Permissions []string `json:"permissions"`
}

// Response is the standard response format
type Response struct {
    Success bool        `json:"success"`
    Data    interface{} `json:"data,omitempty"`
    Error   string      `json:"error,omitempty"`
}

// PluginInit initializes the plugin
func PluginInit(this js.Value, args []js.Value) interface{} {
    info := PluginInfo{
        ID:          "my-wasm-plugin",
        Name:        "My WASM Plugin",
        Version:     "1.0.0",
        Description: "A WASM plugin for the IDE",
        Permissions: []string{"file:read", "file:write"},
    }

    data, _ := json.Marshal(info)
    return js.ValueOf(string(data))
}

// Execute handles commands from the IDE
func Execute(this js.Value, args []js.Value) interface{} {
    if len(args) < 2 {
        return js.ValueOf(jsonError("missing arguments"))
    }

    command := args[0].String()
    paramsStr := args[1].String()

    var params map[string]interface{}
    if err := json.Unmarshal([]byte(paramsStr), &params); err != nil {
        return js.ValueOf(jsonError("invalid parameters"))
    }

    switch command {
    case "hello":
        return js.ValueOf(handleHello(params))
    case "process":
        return js.ValueOf(handleProcess(params))
    default:
        return js.ValueOf(jsonError("unknown command: " + command))
    }
}

// handleHello handles the hello command
func handleHello(params map[string]interface{}) string {
    name, ok := params["name"].(string)
    if !ok {
        return jsonError("name parameter required")
    }

    response := Response{
        Success: true,
        Data: map[string]interface{}{
            "message": "Hello, " + name + "!",
        },
    }

    data, _ := json.Marshal(response)
    return string(data)
}

// handleProcess handles the process command
func handleProcess(params map[string]interface{}) string {
    data, ok := params["data"].(string)
    if !ok {
        return jsonError("data parameter required")
    }

    // Process the data
    result := processData(data)

    response := Response{
        Success: true,
        Data: map[string]interface{}{
            "input":  data,
            "output": result,
        },
    }

    respData, _ := json.Marshal(response)
    return string(respData)
}

// processData is a helper function
func processData(data string) string {
    // Your processing logic here
    return data
}

// jsonError creates an error response
func jsonError(msg string) string {
    response := Response{
        Success: false,
        Error:   msg,
    }
    data, _ := json.Marshal(response)
    return string(data)
}

// Register JavaScript functions
func init() {
    js.Global().Set("PluginInit", js.FuncOf(PluginInit))
    js.Global().Set("Execute", js.FuncOf(Execute))
}

func main() {
    select {} // Block forever
}
```

### Building WASM Plugin

```bash
# Compile to WASM with Go 1.21+
GOOS=js GOARCH=wasm go build -o my-wasm-plugin.wasm .

# Or use TinyGo for smaller size
tinygo build -target=wasm -o my-wasm-plugin.wasm .
```

### WASM Plugin Metadata

**File: plugin.json**

```json
{
  "id": "my-wasm-plugin",
  "name": "My WASM Plugin",
  "version": "1.0.0",
  "author": "Your Name",
  "description": "A WASM plugin",
  "type": "wasm",
  "main": "my-wasm-plugin.wasm",
  "entry": "PluginInit",
  "permissions": [
    "file:read"
  ]
}
```

---

## Plugin Lifecycle

### Load Phase

```
1. Plugin Discovery
   ├─ Scan plugin directories
   ├─ Read plugin.json metadata
   └─ Validate plugin format

2. Permission Check
   ├─ Compare requested permissions
   ├─ Prompt user if needed
   └─ Deny if not granted

3. Plugin Loading
   ├─ Load plugin binary/WASM
   ├─ Initialize runtime
   └─ Call Activate() hook

4. Ready for Commands
   ├─ Plugin listening for requests
   └─ Commands routed to Execute()
```

### Active Phase

```
Plugin Active
├─ Receive WebSocket messages
├─ Execute command handlers
├─ Return results
└─ Listen for deactivation
```

### Unload Phase

```
1. Deactivation Request
   ├─ Call Deactivate() hook
   └─ Allow cleanup

2. Resource Release
   ├─ Close open resources
   ├─ Clear temporary data
   └─ Unload binary/WASM

3. State Cleanup
   ├─ Remove from registry
   ├─ Free memory
   └─ Done
```

---

## Permission System

### Permission Types

Plugins request permissions for system access:

#### File Permissions

```json
{
  "permissions": [
    "file:read",        // Read files in workspace
    "file:write",       // Write files in workspace
    "file:delete",      // Delete files in workspace
    "file:execute"      // Execute files in workspace
  ]
}
```

#### Network Permissions

```json
{
  "permissions": [
    "network:connect",  // Make outbound connections
    "network:listen"    // Listen on ports
  ]
}
```

#### Process Permissions

```json
{
  "permissions": [
    "process:execute",  // Execute external commands
    "process:spawn"     // Spawn subprocesses
  ]
}
```

#### System Permissions

```json
{
  "permissions": [
    "system:env",       // Access environment variables
    "system:time"       // Access system time
  ]
}
```

### Permission Usage

Check permissions before accessing resources:

```go
// Check if plugin has file:read permission
if !hasPermission(plugin, "file:read") {
    return fmt.Errorf("file:read permission required")
}

// Read file
content, err := readFile(path)
```

---

## Plugin API Reference

### IDE Context

Plugins receive IDE context:

```go
type IDEContext struct {
    Workspace   string                 // Workspace directory
    EditorState map[string]interface{} // Current editor state
    Config      map[string]interface{} // IDE configuration
}
```

### Common APIs

#### File API

```go
// Read file
func ReadFile(path string) ([]byte, error)

// Write file
func WriteFile(path string, content []byte) error

// List directory
func ListDir(path string) ([]string, error)

// Watch file
func WatchFile(path string, callback func(event string)) error
```

#### Editor API

```go
// Get current file
func GetCurrentFile() (string, error)

// Get cursor position
func GetCursorPosition() (line int, column int, error)

// Get selected text
func GetSelectedText() (string, error)

// Replace text
func ReplaceText(range Range, newText string) error

// Insert text
func InsertText(line, column int, text string) error
```

#### LSP API

```go
// Get completions
func GetCompletions(uri string, line, column int) ([]Completion, error)

// Get hover info
func GetHover(uri string, line, column int) (string, error)

// Go to definition
func GoToDefinition(uri string, line, column int) (Location, error)

// Find references
func FindReferences(uri string, line, column int) ([]Location, error)
```

#### Terminal API

```go
// Execute command
func Execute(command string) (stdout, stderr string, exitCode int, error)

// Execute with timeout
func ExecuteTimeout(command string, timeout time.Duration) (stdout, stderr string, exitCode int, error)

// Get environment
func GetEnv(key string) (string, error)
```

#### Notification API

```go
// Show message
func ShowMessage(level string, message string) error

// Show error
func ShowError(message string) error

// Show info
func ShowInfo(message string) error

// Show warning
func ShowWarning(message string) error
```

---

## Examples

### Example 1: Code Formatter Plugin

**File: main.go**

```go
package main

import (
    "encoding/json"
    "fmt"
    "go/format"
)

type FormatterPlugin struct {
    name    string
    version string
}

func NewFormatterPlugin() *FormatterPlugin {
    return &FormatterPlugin{
        name:    "go-formatter",
        version: "1.0.0",
    }
}

func (p *FormatterPlugin) Name() string {
    return p.name
}

func (p *FormatterPlugin) Version() string {
    return p.version
}

func (p *FormatterPlugin) Description() string {
    return "Formats Go code using gofmt"
}

func (p *FormatterPlugin) Activate() error {
    fmt.Println("Go formatter plugin activated")
    return nil
}

func (p *FormatterPlugin) Deactivate() error {
    fmt.Println("Go formatter plugin deactivated")
    return nil
}

func (p *FormatterPlugin) Execute(command string, params map[string]interface{}) (interface{}, error) {
    switch command {
    case "format":
        code, ok := params["code"].(string)
        if !ok {
            return nil, fmt.Errorf("code parameter required")
        }

        formatted, err := format.Source([]byte(code))
        if err != nil {
            return nil, fmt.Errorf("format error: %w", err)
        }

        return map[string]interface{}{
            "formatted": string(formatted),
        }, nil

    default:
        return nil, fmt.Errorf("unknown command: %s", command)
    }
}

var Plugin *FormatterPlugin

func init() {
    Plugin = NewFormatterPlugin()
}
```

### Example 2: Custom Linter Plugin

**File: main.go**

```go
package main

import (
    "fmt"
    "go/ast"
    "go/parser"
    "go/token"
    "strings"
)

type LinterPlugin struct {
    name    string
    version string
}

func (p *LinterPlugin) Name() string {
    return "custom-linter"
}

func (p *LinterPlugin) Version() string {
    return "1.0.0"
}

func (p *LinterPlugin) Description() string {
    return "Custom Go code linter"
}

func (p *LinterPlugin) Activate() error {
    return nil
}

func (p *LinterPlugin) Deactivate() error {
    return nil
}

func (p *LinterPlugin) Execute(command string, params map[string]interface{}) (interface{}, error) {
    switch command {
    case "lint":
        code, ok := params["code"].(string)
        if !ok {
            return nil, fmt.Errorf("code parameter required")
        }

        issues, err := p.lintCode(code)
        if err != nil {
            return nil, fmt.Errorf("lint error: %w", err)
        }

        return map[string]interface{}{
            "issues": issues,
            "count":  len(issues),
        }, nil

    default:
        return nil, fmt.Errorf("unknown command: %s", command)
    }
}

func (p *LinterPlugin) lintCode(code string) ([]map[string]interface{}, error) {
    fset := token.NewFileSet()
    tree, err := parser.ParseFile(fset, "src.go", code, parser.AllErrors)
    if err != nil {
        return nil, fmt.Errorf("parse error: %w", err)
    }

    issues := []map[string]interface{}{}

    // Check for unexported functions with doc comments
    for _, decl := range tree.Decls {
        if fn, ok := decl.(*ast.FuncDecl); ok {
            if fn.Name.IsExported() && fn.Doc == nil {
                issues = append(issues, map[string]interface{}{
                    "line":     fset.Position(fn.Pos()).Line,
                    "column":   fset.Position(fn.Pos()).Column,
                    "severity": "warning",
                    "message":  fmt.Sprintf("exported function %s should have comment", fn.Name.Name),
                })
            }
        }
    }

    return issues, nil
}

var Plugin *LinterPlugin

func init() {
    Plugin = &LinterPlugin{
        name:    "custom-linter",
        version: "1.0.0",
    }
}
```

---

## Testing

### Unit Tests

**File: handler_test.go**

```go
package main

import (
    "testing"
)

func TestHello(t *testing.T) {
    plugin := NewFormatterPlugin()

    params := map[string]interface{}{
        "code": "package main\nfunc main(){fmt.Println(\"hello\")}",
    }

    result, err := plugin.Execute("format", params)
    if err != nil {
        t.Fatalf("Execute failed: %v", err)
    }

    resultMap := result.(map[string]interface{})
    formatted := resultMap["formatted"].(string)

    if formatted == "" {
        t.Fatal("formatted code is empty")
    }

    t.Logf("Formatted:\n%s", formatted)
}

func TestActivate(t *testing.T) {
    plugin := NewFormatterPlugin()

    if err := plugin.Activate(); err != nil {
        t.Fatalf("Activate failed: %v", err)
    }

    if plugin.Name() != "go-formatter" {
        t.Fatalf("Expected name 'go-formatter', got '%s'", plugin.Name())
    }
}
```

**Run tests:**

```bash
go test -v ./...
```

---

## Distribution

### Creating Distribution Package

```bash
# Build plugin
go build -buildmode=plugin -o my-plugin.so .

# Create distribution directory
mkdir my-plugin-v1.0.0
cp my-plugin.so my-plugin-v1.0.0/
cp plugin.json my-plugin-v1.0.0/
cp README.md my-plugin-v1.0.0/
cp LICENSE my-plugin-v1.0.0/

# Create tarball
tar -czf my-plugin-v1.0.0.tar.gz my-plugin-v1.0.0/

# Or create zip
zip -r my-plugin-v1.0.0.zip my-plugin-v1.0.0/
```

### Plugin Manifest

**manifest.json** (for plugin marketplace):

```json
{
  "id": "my-plugin",
  "name": "My Plugin",
  "version": "1.0.0",
  "author": "Your Name",
  "description": "Plugin description",
  "homepage": "https://github.com/yourorg/my-plugin",
  "download": "https://github.com/yourorg/my-plugin/releases/download/v1.0.0/my-plugin-v1.0.0.tar.gz",
  "checksums": {
    "my-plugin.so": "sha256:abc123..."
  }
}
```

---

**Plugin Development Guide Version**: 1.0.0
**Last Updated**: 2025-10-29
