# Testing Guide

## Build Verification

The project has been successfully built and verified:

```bash
$ go mod tidy
✓ Dependencies downloaded

$ go build -o bin/ide ./cmd/ide
✓ Binary built successfully (12MB)
```

## Component Status

### ✅ Core Components
- [x] HTTP Server with WebSocket support
- [x] LSP Manager (gopls integration)
- [x] DAP Manager (Delve integration placeholder)
- [x] Git Manager (go-git)
- [x] File Manager (fsnotify)
- [x] Terminal Manager
- [x] Plugin Manager

### ✅ Build System
- [x] Makefile with build, test, docker targets
- [x] Dockerfile for containerized deployment
- [x] Go modules configuration
- [x] Example plugins

## Running Tests

### Build the Project
```bash
make build
```

### Run the Server
```bash
# Start in current directory
./bin/ide --workspace=.

# Start in specific workspace
./bin/ide --workspace=/path/to/your/go/project

# Verbose logging
./bin/ide -v
```

### Test WebSocket Connection
```bash
# In browser console:
const ws = new WebSocket('ws://localhost:3000/ws');
ws.onopen = () => console.log('Connected');
ws.onmessage = (e) => console.log('Received:', JSON.parse(e.data));

// Send LSP request
ws.send(JSON.stringify({
  id: 'req-1',
  type: 'lsp',
  command: 'completion',
  params: { uri: 'file:///workspace/main.go', line: 10, column: 5 }
}));
```

### Test Individual Components

#### LSP (Language Server Protocol)
```bash
# Requires gopls installed
go install golang.org/x/tools/gopls@latest

# LSP will auto-start when server runs
# Test completion, hover, definition via WebSocket
```

#### Git Operations
```bash
# In a Git repository
./bin/ide --workspace=/path/to/git/repo

# WebSocket requests:
# - status: Check repository status
# - branches: List branches
# - commit: Create commit
# - log: View commit history
```

#### File Operations
```bash
# WebSocket requests:
# - tree: Get file tree
# - read: Read file content
# - write: Write file content
# - search: Search in files
```

## Dependencies

### Required
- Go 1.21+
- Git (for Git features)

### Optional
- gopls (for LSP features): `go install golang.org/x/tools/gopls@latest`
- delve (for debugging): `go install github.com/go-delve/delve/cmd/dlv@latest`

## Known Limitations

1. **DAP Integration**: Mock implementation included. Full Delve integration requires additional work.
2. **Frontend**: Placeholder HTML interface. Build custom frontend or use API directly.
3. **Plugins**: Plugin system framework in place, WASM integration requires additional work.

## Architecture Verification

All core architectural components are implemented:

1. **WebSocket Communication** ✓
   - Bidirectional message routing
   - JSON-based protocol
   - Multi-client support

2. **LSP Integration** ✓
   - gopls process management
   - JSON-RPC communication
   - LSP protocol implementation
   - Completion, hover, definition, references

3. **DAP Framework** ✓
   - Debug session management
   - Breakpoint handling (mock)
   - Stack trace (mock)
   - Ready for Delve integration

4. **Git Integration** ✓
   - Pure Go implementation (go-git)
   - Status, diff, commit, branches
   - No external git binary required

5. **File Management** ✓
   - CRUD operations
   - File watching (fsnotify)
   - Project-wide search
   - Path traversal protection

6. **Terminal** ✓
   - Command execution
   - Session management
   - Output streaming

7. **Plugin System** ✓
   - Plugin loading/unloading
   - Permission system
   - Example plugins included

## Next Steps

### To Enable Full DAP Support
1. Integrate Delve via DAP protocol
2. Implement breakpoint synchronization
3. Add variable inspection
4. Support goroutine debugging

### To Build Production Frontend
1. Use CodeMirror 6 for editor
2. Use xterm.js for terminal
3. Build with Vite
4. Embed in Go binary

### To Enable Plugin WASM
1. Integrate wazero runtime
2. Define WASM host functions
3. Build plugin SDK
4. Create plugin marketplace
