package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"path/filepath"

	"github.com/yourorg/go-ide/internal/server"
)

// Note: Static files would be embedded here in production
// For now, we'll use a placeholder filesystem
var staticFiles server.StaticFS

func main() {
	port := flag.String("port", "3000", "Server port")
	workspace := flag.String("workspace", ".", "Workspace directory")
	verbose := flag.Bool("v", false, "Verbose logging")
	flag.Parse()

	// Resolve absolute workspace path
	absWorkspace, err := filepath.Abs(*workspace)
	if err != nil {
		log.Fatalf("Failed to resolve workspace: %v", err)
	}

	// Verify workspace exists
	if _, err := os.Stat(absWorkspace); os.IsNotExist(err) {
		log.Fatalf("Workspace directory does not exist: %s", absWorkspace)
	}

	// Create IDE server
	srv := server.New(&server.Config{
		Port:        *port,
		Workspace:   absWorkspace,
		StaticFiles: staticFiles,
		Verbose:     *verbose,
	})

	fmt.Printf("\n🚀 Go IDE Platform Starting...\n")
	fmt.Printf("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n")
	fmt.Printf("📂 Workspace: %s\n", absWorkspace)
	fmt.Printf("🌐 Server:    http://localhost:%s\n", *port)
	fmt.Printf("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n")
	fmt.Printf("Features:\n")
	fmt.Printf("  ✓ LSP Integration (gopls)\n")
	fmt.Printf("  ✓ DAP Debugger (delve)\n")
	fmt.Printf("  ✓ Git Integration\n")
	fmt.Printf("  ✓ File Watcher\n")
	fmt.Printf("  ✓ Plugin System\n")
	fmt.Printf("  ✓ Integrated Terminal\n\n")
	fmt.Printf("Press Ctrl+C to stop the server.\n\n")

	if err := srv.Start(); err != nil {
		log.Fatalf("Server failed: %v", err)
	}
}
