package dap

import (
	"encoding/json"
	"fmt"

	"github.com/rs/zerolog/log"
)

// Manager handles Debug Adapter Protocol communication with Delve
type Manager struct {
	sessions map[string]*DebugSession
}

type DebugSession struct {
	ID        string
	Program   string
	Args      []string
	Running   bool
	Breakpoints map[string][]Breakpoint
}

type Breakpoint struct {
	ID        int    `json:"id"`
	File      string `json:"file"`
	Line      int    `json:"line"`
	Condition string `json:"condition,omitempty"`
	Verified  bool   `json:"verified"`
}

func NewManager() *Manager {
	return &Manager{
		sessions: make(map[string]*DebugSession),
	}
}

// Handle processes DAP commands from WebSocket
func (m *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling DAP command")

	switch command {
	case "launch":
		return m.handleLaunch(params)
	case "setBreakpoint":
		return m.handleSetBreakpoint(params)
	case "continue":
		return m.handleContinue(params)
	case "stepOver":
		return m.handleStepOver(params)
	case "stepInto":
		return m.handleStepInto(params)
	case "stepOut":
		return m.handleStepOut(params)
	case "pause":
		return m.handlePause(params)
	case "stop":
		return m.handleStop(params)
	case "stackTrace":
		return m.handleStackTrace(params)
	case "variables":
		return m.handleVariables(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown DAP command: %s", command),
		}
	}
}

func (m *Manager) handleLaunch(params json.RawMessage) interface{} {
	var p struct {
		Program string   `json:"program"`
		Args    []string `json:"args"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	sessionID := "session-1" // Simplified for example
	m.sessions[sessionID] = &DebugSession{
		ID:          sessionID,
		Program:     p.Program,
		Args:        p.Args,
		Running:     false,
		Breakpoints: make(map[string][]Breakpoint),
	}

	log.Info().Str("program", p.Program).Msg("Debug session launched")

	return map[string]interface{}{
		"sessionId": sessionID,
		"status":    "ready",
		"message":   "Debug session created. Note: Full DAP requires delve integration",
	}
}

func (m *Manager) handleSetBreakpoint(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
		File      string `json:"file"`
		Line      int    `json:"line"`
		Condition string `json:"condition"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	session, ok := m.sessions[p.SessionID]
	if !ok {
		return map[string]interface{}{"error": "session not found"}
	}

	bp := Breakpoint{
		ID:        len(session.Breakpoints[p.File]) + 1,
		File:      p.File,
		Line:      p.Line,
		Condition: p.Condition,
		Verified:  true,
	}

	session.Breakpoints[p.File] = append(session.Breakpoints[p.File], bp)

	log.Info().Str("file", p.File).Int("line", p.Line).Msg("Breakpoint set")

	return map[string]interface{}{
		"breakpoint": bp,
		"message":    "Breakpoint set (mock implementation)",
	}
}

func (m *Manager) handleContinue(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	session, ok := m.sessions[p.SessionID]
	if !ok {
		return map[string]interface{}{"error": "session not found"}
	}

	session.Running = true
	log.Info().Msg("Program continued")

	return map[string]interface{}{
		"status":  "running",
		"message": "Program continued (mock implementation)",
	}
}

func (m *Manager) handleStepOver(params json.RawMessage) interface{} {
	return map[string]interface{}{
		"status":  "paused",
		"message": "Step over executed (mock implementation)",
	}
}

func (m *Manager) handleStepInto(params json.RawMessage) interface{} {
	return map[string]interface{}{
		"status":  "paused",
		"message": "Step into executed (mock implementation)",
	}
}

func (m *Manager) handleStepOut(params json.RawMessage) interface{} {
	return map[string]interface{}{
		"status":  "paused",
		"message": "Step out executed (mock implementation)",
	}
}

func (m *Manager) handlePause(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	session, ok := m.sessions[p.SessionID]
	if !ok {
		return map[string]interface{}{"error": "session not found"}
	}

	session.Running = false
	log.Info().Msg("Program paused")

	return map[string]interface{}{
		"status":  "paused",
		"message": "Program paused (mock implementation)",
	}
}

func (m *Manager) handleStop(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	delete(m.sessions, p.SessionID)
	log.Info().Msg("Debug session stopped")

	return map[string]interface{}{
		"status":  "stopped",
		"message": "Debug session stopped",
	}
}

func (m *Manager) handleStackTrace(params json.RawMessage) interface{} {
	return map[string]interface{}{
		"stackFrames": []map[string]interface{}{
			{
				"id":     1,
				"name":   "main.main",
				"file":   "main.go",
				"line":   42,
				"column": 1,
			},
		},
		"message": "Stack trace (mock implementation)",
	}
}

func (m *Manager) handleVariables(params json.RawMessage) interface{} {
	return map[string]interface{}{
		"variables": []map[string]interface{}{
			{
				"name":  "x",
				"value": "42",
				"type":  "int",
			},
			{
				"name":  "msg",
				"value": "\"hello\"",
				"type":  "string",
			},
		},
		"message": "Variables (mock implementation)",
	}
}
