package files

import (
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"strings"

	"github.com/fsnotify/fsnotify"
	"github.com/rs/zerolog/log"
)

// Manager handles file operations and watching
type Manager struct {
	root    string
	watcher *fsnotify.Watcher
	events  chan FileEvent
}

type FileEvent struct {
	Type string `json:"type"` // "create", "modify", "delete"
	Path string `json:"path"`
}

type FileNode struct {
	Name     string      `json:"name"`
	Path     string      `json:"path"`
	IsDir    bool        `json:"isDir"`
	Size     int64       `json:"size,omitempty"`
	Children []*FileNode `json:"children,omitempty"`
}

type SearchResult struct {
	File   string `json:"file"`
	Line   int    `json:"line"`
	Column int    `json:"column"`
	Text   string `json:"text"`
}

func NewManager(root string) (*Manager, error) {
	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		return nil, fmt.Errorf("failed to create file watcher: %w", err)
	}

	fm := &Manager{
		root:    root,
		watcher: watcher,
		events:  make(chan FileEvent, 100),
	}

	// Watch root directory
	if err := watcher.Add(root); err != nil {
		return nil, fmt.Errorf("failed to watch directory: %w", err)
	}

	// Start event processor
	go fm.processEvents()

	log.Info().Str("root", root).Msg("File manager initialized")

	return fm, nil
}

func (fm *Manager) processEvents() {
	for {
		select {
		case event, ok := <-fm.watcher.Events:
			if !ok {
				return
			}

			var eventType string
			switch {
			case event.Op&fsnotify.Create == fsnotify.Create:
				eventType = "create"
			case event.Op&fsnotify.Write == fsnotify.Write:
				eventType = "modify"
			case event.Op&fsnotify.Remove == fsnotify.Remove:
				eventType = "delete"
			default:
				continue
			}

			fm.events <- FileEvent{
				Type: eventType,
				Path: strings.TrimPrefix(event.Name, fm.root+"/"),
			}

		case err, ok := <-fm.watcher.Errors:
			if !ok {
				return
			}
			log.Error().Err(err).Msg("File watcher error")
		}
	}
}

// Handle processes file commands from WebSocket
func (fm *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling file command")

	switch command {
	case "read":
		return fm.handleRead(params)
	case "write":
		return fm.handleWrite(params)
	case "tree":
		return fm.handleTree(params)
	case "search":
		return fm.handleSearch(params)
	case "delete":
		return fm.handleDelete(params)
	case "rename":
		return fm.handleRename(params)
	case "mkdir":
		return fm.handleMkdir(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown file command: %s", command),
		}
	}
}

func (fm *Manager) handleRead(params json.RawMessage) interface{} {
	var p struct {
		Path string `json:"path"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	fullPath := filepath.Join(fm.root, p.Path)

	// Security: prevent path traversal
	if !strings.HasPrefix(fullPath, fm.root) {
		return map[string]interface{}{"error": "access denied: path outside workspace"}
	}

	content, err := os.ReadFile(fullPath)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{
		"path":    p.Path,
		"content": string(content),
	}
}

func (fm *Manager) handleWrite(params json.RawMessage) interface{} {
	var p struct {
		Path    string `json:"path"`
		Content string `json:"content"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	fullPath := filepath.Join(fm.root, p.Path)

	// Security check
	if !strings.HasPrefix(fullPath, fm.root) {
		return map[string]interface{}{"error": "access denied: path outside workspace"}
	}

	// Create directory if needed
	dir := filepath.Dir(fullPath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	if err := os.WriteFile(fullPath, []byte(p.Content), 0644); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("path", p.Path).Msg("File written")

	return map[string]interface{}{
		"path":    p.Path,
		"message": "File written successfully",
	}
}

func (fm *Manager) handleTree(params json.RawMessage) interface{} {
	tree, err := fm.buildTree(fm.root)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{
		"tree": tree,
	}
}

func (fm *Manager) buildTree(path string) (*FileNode, error) {
	info, err := os.Stat(path)
	if err != nil {
		return nil, err
	}

	node := &FileNode{
		Name:  filepath.Base(path),
		Path:  strings.TrimPrefix(path, fm.root+"/"),
		IsDir: info.IsDir(),
		Size:  info.Size(),
	}

	if info.IsDir() {
		entries, err := os.ReadDir(path)
		if err != nil {
			return nil, err
		}

		for _, entry := range entries {
			// Skip hidden files, .git, and common build directories
			name := entry.Name()
			if strings.HasPrefix(name, ".") ||
			   name == "node_modules" ||
			   name == "vendor" ||
			   name == "dist" ||
			   name == "build" {
				continue
			}

			child, err := fm.buildTree(filepath.Join(path, name))
			if err != nil {
				continue
			}

			node.Children = append(node.Children, child)
		}
	}

	return node, nil
}

func (fm *Manager) handleSearch(params json.RawMessage) interface{} {
	var p struct {
		Query string `json:"query"`
		Regex bool   `json:"regex"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	results, err := fm.search(p.Query, p.Regex)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{
		"results": results,
		"total":   len(results),
	}
}

func (fm *Manager) search(query string, useRegex bool) ([]SearchResult, error) {
	var results []SearchResult
	var pattern *regexp.Regexp
	var err error

	if useRegex {
		pattern, err = regexp.Compile(query)
		if err != nil {
			return nil, fmt.Errorf("invalid regex: %w", err)
		}
	}

	err = filepath.Walk(fm.root, func(path string, info os.FileInfo, err error) error {
		if err != nil || info.IsDir() {
			return err
		}

		// Skip binary and large files
		if info.Size() > 10*1024*1024 { // 10MB
			return nil
		}

		content, err := os.ReadFile(path)
		if err != nil {
			return nil
		}

		lines := strings.Split(string(content), "\n")
		for i, line := range lines {
			matched := false

			if useRegex {
				matched = pattern.MatchString(line)
			} else {
				matched = strings.Contains(line, query)
			}

			if matched {
				col := 0
				if !useRegex {
					col = strings.Index(line, query)
				}

				results = append(results, SearchResult{
					File:   strings.TrimPrefix(path, fm.root+"/"),
					Line:   i + 1,
					Column: col + 1,
					Text:   line,
				})

				// Limit results per file
				if len(results) > 1000 {
					return filepath.SkipDir
				}
			}
		}

		return nil
	})

	return results, err
}

func (fm *Manager) handleDelete(params json.RawMessage) interface{} {
	var p struct {
		Path string `json:"path"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	fullPath := filepath.Join(fm.root, p.Path)

	// Security check
	if !strings.HasPrefix(fullPath, fm.root) {
		return map[string]interface{}{"error": "access denied: path outside workspace"}
	}

	if err := os.RemoveAll(fullPath); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("path", p.Path).Msg("File deleted")

	return map[string]interface{}{
		"path":    p.Path,
		"message": "File deleted successfully",
	}
}

func (fm *Manager) handleRename(params json.RawMessage) interface{} {
	var p struct {
		OldPath string `json:"oldPath"`
		NewPath string `json:"newPath"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	oldFullPath := filepath.Join(fm.root, p.OldPath)
	newFullPath := filepath.Join(fm.root, p.NewPath)

	// Security checks
	if !strings.HasPrefix(oldFullPath, fm.root) || !strings.HasPrefix(newFullPath, fm.root) {
		return map[string]interface{}{"error": "access denied: path outside workspace"}
	}

	if err := os.Rename(oldFullPath, newFullPath); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("from", p.OldPath).Str("to", p.NewPath).Msg("File renamed")

	return map[string]interface{}{
		"oldPath": p.OldPath,
		"newPath": p.NewPath,
		"message": "File renamed successfully",
	}
}

func (fm *Manager) handleMkdir(params json.RawMessage) interface{} {
	var p struct {
		Path string `json:"path"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	fullPath := filepath.Join(fm.root, p.Path)

	// Security check
	if !strings.HasPrefix(fullPath, fm.root) {
		return map[string]interface{}{"error": "access denied: path outside workspace"}
	}

	if err := os.MkdirAll(fullPath, 0755); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("path", p.Path).Msg("Directory created")

	return map[string]interface{}{
		"path":    p.Path,
		"message": "Directory created successfully",
	}
}
