package git

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/go-git/go-git/v5"
	"github.com/go-git/go-git/v5/plumbing"
	"github.com/go-git/go-git/v5/plumbing/object"
	"github.com/rs/zerolog/log"
)

// Manager handles Git operations using go-git
type Manager struct {
	repo *git.Repository
	path string
}

func NewManager(workspacePath string) (*Manager, error) {
	repo, err := git.PlainOpen(workspacePath)
	if err != nil {
		return nil, fmt.Errorf("not a git repository: %w", err)
	}

	log.Info().Str("path", workspacePath).Msg("Git repository opened")

	return &Manager{
		repo: repo,
		path: workspacePath,
	}, nil
}

// Handle processes Git commands from WebSocket
func (m *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling Git command")

	switch command {
	case "status":
		return m.handleStatus(params)
	case "diff":
		return m.handleDiff(params)
	case "commit":
		return m.handleCommit(params)
	case "branches":
		return m.handleBranches(params)
	case "checkout":
		return m.handleCheckout(params)
	case "log":
		return m.handleLog(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown Git command: %s", command),
		}
	}
}

func (m *Manager) handleStatus(params json.RawMessage) interface{} {
	worktree, err := m.repo.Worktree()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	status, err := worktree.Status()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	modified := []string{}
	staged := []string{}
	untracked := []string{}

	for path, stat := range status {
		switch stat.Worktree {
		case git.Modified:
			modified = append(modified, path)
		case git.Untracked:
			untracked = append(untracked, path)
		}

		if stat.Staging != git.Unmodified {
			staged = append(staged, path)
		}
	}

	return map[string]interface{}{
		"modified":   modified,
		"staged":     staged,
		"untracked":  untracked,
		"clean":      len(modified) == 0 && len(staged) == 0 && len(untracked) == 0,
	}
}

func (m *Manager) handleDiff(params json.RawMessage) interface{} {
	var p struct {
		File string `json:"file"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	// Simplified diff implementation
	head, err := m.repo.Head()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	commit, err := m.repo.CommitObject(head.Hash())
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{
		"file":    p.File,
		"commit":  commit.Hash.String()[:7],
		"message": "Diff available (simplified implementation)",
	}
}

func (m *Manager) handleCommit(params json.RawMessage) interface{} {
	var p struct {
		Message string   `json:"message"`
		Files   []string `json:"files"`
		Author  string   `json:"author"`
		Email   string   `json:"email"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	worktree, err := m.repo.Worktree()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	// Stage files
	for _, file := range p.Files {
		if _, err := worktree.Add(file); err != nil {
			return map[string]interface{}{"error": fmt.Sprintf("failed to add %s: %v", file, err)}
		}
	}

	// Commit
	author := "IDE User"
	email := "user@example.com"
	if p.Author != "" {
		author = p.Author
	}
	if p.Email != "" {
		email = p.Email
	}

	hash, err := worktree.Commit(p.Message, &git.CommitOptions{
		Author: &object.Signature{
			Name:  author,
			Email: email,
			When:  time.Now(),
		},
	})

	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("hash", hash.String()[:7]).Str("message", p.Message).Msg("Committed changes")

	return map[string]interface{}{
		"hash":    hash.String(),
		"message": p.Message,
		"author":  author,
	}
}

func (m *Manager) handleBranches(params json.RawMessage) interface{} {
	branches, err := m.repo.Branches()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	var names []string
	branches.ForEach(func(ref *plumbing.Reference) error {
		names = append(names, ref.Name().Short())
		return nil
	})

	// Get current branch
	head, err := m.repo.Head()
	var current string
	if err == nil {
		current = head.Name().Short()
	}

	return map[string]interface{}{
		"branches": names,
		"current":  current,
	}
}

func (m *Manager) handleCheckout(params json.RawMessage) interface{} {
	var p struct {
		Branch string `json:"branch"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	worktree, err := m.repo.Worktree()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	err = worktree.Checkout(&git.CheckoutOptions{
		Branch: plumbing.ReferenceName("refs/heads/" + p.Branch),
	})

	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	log.Info().Str("branch", p.Branch).Msg("Checked out branch")

	return map[string]interface{}{
		"branch":  p.Branch,
		"message": fmt.Sprintf("Switched to branch '%s'", p.Branch),
	}
}

func (m *Manager) handleLog(params json.RawMessage) interface{} {
	var p struct {
		Limit int `json:"limit"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	if p.Limit == 0 {
		p.Limit = 10
	}

	head, err := m.repo.Head()
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	commitIter, err := m.repo.Log(&git.LogOptions{From: head.Hash()})
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	commits := []map[string]interface{}{}
	count := 0

	commitIter.ForEach(func(c *object.Commit) error {
		if count >= p.Limit {
			return fmt.Errorf("limit reached")
		}

		commits = append(commits, map[string]interface{}{
			"hash":    c.Hash.String()[:7],
			"message": c.Message,
			"author":  c.Author.Name,
			"email":   c.Author.Email,
			"date":    c.Author.When.Format(time.RFC3339),
		})

		count++
		return nil
	})

	return map[string]interface{}{
		"commits": commits,
		"total":   count,
	}
}
