package lsp

import (
	"bufio"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"os/exec"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/rs/zerolog/log"
)

// Manager handles Language Server Protocol communication with gopls
type Manager struct {
	workspace string
	process   *exec.Cmd
	stdin     io.WriteCloser
	stdout    io.ReadCloser
	requests  map[int]chan *Response
	mu        sync.Mutex
	nextID    int
	ctx       context.Context
	cancel    context.CancelFunc
}

type Request struct {
	JSONRPC string      `json:"jsonrpc"`
	ID      int         `json:"id,omitempty"`
	Method  string      `json:"method"`
	Params  interface{} `json:"params,omitempty"`
}

type Response struct {
	JSONRPC string          `json:"jsonrpc"`
	ID      int             `json:"id,omitempty"`
	Result  json.RawMessage `json:"result,omitempty"`
	Error   *RPCError       `json:"error,omitempty"`
}

type RPCError struct {
	Code    int    `json:"code"`
	Message string `json:"message"`
}

func NewManager(workspace string) (*Manager, error) {
	ctx, cancel := context.WithCancel(context.Background())

	m := &Manager{
		workspace: workspace,
		requests:  make(map[int]chan *Response),
		ctx:       ctx,
		cancel:    cancel,
	}

	if err := m.start(); err != nil {
		cancel()
		return nil, err
	}

	log.Info().Msg("LSP manager started with gopls")
	return m, nil
}

func (m *Manager) start() error {
	// Find gopls binary
	goplsPath, err := exec.LookPath("gopls")
	if err != nil {
		return fmt.Errorf("gopls not found in PATH: %w (install with: go install golang.org/x/tools/gopls@latest)", err)
	}

	log.Debug().Str("path", goplsPath).Msg("Found gopls")

	// Start gopls in stdio mode
	m.process = exec.CommandContext(m.ctx, goplsPath, "-mode=stdio")

	stdin, err := m.process.StdinPipe()
	if err != nil {
		return fmt.Errorf("failed to create stdin pipe: %w", err)
	}
	m.stdin = stdin

	stdout, err := m.process.StdoutPipe()
	if err != nil {
		return fmt.Errorf("failed to create stdout pipe: %w", err)
	}
	m.stdout = stdout

	if err := m.process.Start(); err != nil {
		return fmt.Errorf("failed to start gopls: %w", err)
	}

	// Read responses in background
	go m.readResponses()

	// Initialize LSP
	if err := m.initialize(); err != nil {
		return fmt.Errorf("failed to initialize LSP: %w", err)
	}

	return nil
}

func (m *Manager) initialize() error {
	req := &Request{
		JSONRPC: "2.0",
		Method:  "initialize",
		Params: map[string]interface{}{
			"processId": nil,
			"rootUri":   "file://" + m.workspace,
			"capabilities": map[string]interface{}{
				"textDocument": map[string]interface{}{
					"completion": map[string]interface{}{
						"completionItem": map[string]interface{}{
							"snippetSupport": true,
						},
					},
					"hover": map[string]interface{}{
						"contentFormat": []string{"markdown", "plaintext"},
					},
				},
			},
		},
	}

	resp, err := m.sendRequest(req)
	if err != nil {
		return err
	}

	if resp.Error != nil {
		return fmt.Errorf("initialize error: %s", resp.Error.Message)
	}

	// Send initialized notification
	if err := m.sendNotification(&Request{
		JSONRPC: "2.0",
		Method:  "initialized",
		Params:  map[string]interface{}{},
	}); err != nil {
		return err
	}

	log.Info().Msg("LSP initialized successfully")
	return nil
}

func (m *Manager) sendRequest(req *Request) (*Response, error) {
	m.mu.Lock()
	id := m.nextID
	m.nextID++
	req.ID = id
	respChan := make(chan *Response, 1)
	m.requests[id] = respChan
	m.mu.Unlock()

	// Encode request
	data, err := json.Marshal(req)
	if err != nil {
		return nil, err
	}

	// Send with Content-Length header (LSP protocol)
	header := fmt.Sprintf("Content-Length: %d\r\n\r\n", len(data))
	if _, err := m.stdin.Write([]byte(header)); err != nil {
		return nil, err
	}
	if _, err := m.stdin.Write(data); err != nil {
		return nil, err
	}

	log.Debug().Int("id", id).Str("method", req.Method).Msg("Sent LSP request")

	// Wait for response (with timeout)
	ctx, cancel := context.WithTimeout(m.ctx, 10*time.Second)
	defer cancel()

	select {
	case resp := <-respChan:
		return resp, nil
	case <-ctx.Done():
		m.mu.Lock()
		delete(m.requests, id)
		m.mu.Unlock()
		return nil, fmt.Errorf("request timeout")
	}
}

func (m *Manager) sendNotification(req *Request) error {
	data, err := json.Marshal(req)
	if err != nil {
		return err
	}

	header := fmt.Sprintf("Content-Length: %d\r\n\r\n", len(data))
	if _, err := m.stdin.Write([]byte(header)); err != nil {
		return err
	}
	_, err = m.stdin.Write(data)
	return err
}

func (m *Manager) readResponses() {
	reader := bufio.NewReader(m.stdout)

	for {
		// Read Content-Length header
		var contentLength int
		for {
			line, err := reader.ReadString('\n')
			if err != nil {
				if err != io.EOF {
					log.Error().Err(err).Msg("Failed to read LSP response header")
				}
				return
			}

			line = strings.TrimSpace(line)
			if line == "" {
				break // End of headers
			}

			if strings.HasPrefix(line, "Content-Length:") {
				parts := strings.Split(line, ":")
				if len(parts) == 2 {
					contentLength, _ = strconv.Atoi(strings.TrimSpace(parts[1]))
				}
			}
		}

		if contentLength == 0 {
			continue
		}

		// Read body
		body := make([]byte, contentLength)
		if _, err := io.ReadFull(reader, body); err != nil {
			log.Error().Err(err).Msg("Failed to read LSP response body")
			return
		}

		// Parse response
		var resp Response
		if err := json.Unmarshal(body, &resp); err != nil {
			log.Error().Err(err).Msg("Failed to parse LSP response")
			continue
		}

		// Route to waiting request
		m.mu.Lock()
		if ch, ok := m.requests[resp.ID]; ok {
			delete(m.requests, resp.ID)
			m.mu.Unlock()
			ch <- &resp
		} else {
			m.mu.Unlock()
			log.Debug().Int("id", resp.ID).Msg("Received response for unknown request")
		}
	}
}

// Handle processes LSP commands from WebSocket
func (m *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling LSP command")

	switch command {
	case "completion":
		return m.handleCompletion(params)
	case "hover":
		return m.handleHover(params)
	case "definition":
		return m.handleDefinition(params)
	case "references":
		return m.handleReferences(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown LSP command: %s", command),
		}
	}
}

func (m *Manager) handleCompletion(params json.RawMessage) interface{} {
	var p struct {
		URI      string `json:"uri"`
		Line     int    `json:"line"`
		Column   int    `json:"column"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	req := &Request{
		JSONRPC: "2.0",
		Method:  "textDocument/completion",
		Params: map[string]interface{}{
			"textDocument": map[string]string{"uri": p.URI},
			"position": map[string]int{
				"line":      p.Line,
				"character": p.Column,
			},
		},
	}

	resp, err := m.sendRequest(req)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{"result": resp.Result}
}

func (m *Manager) handleHover(params json.RawMessage) interface{} {
	var p struct {
		URI    string `json:"uri"`
		Line   int    `json:"line"`
		Column int    `json:"column"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	req := &Request{
		JSONRPC: "2.0",
		Method:  "textDocument/hover",
		Params: map[string]interface{}{
			"textDocument": map[string]string{"uri": p.URI},
			"position": map[string]int{
				"line":      p.Line,
				"character": p.Column,
			},
		},
	}

	resp, err := m.sendRequest(req)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{"result": resp.Result}
}

func (m *Manager) handleDefinition(params json.RawMessage) interface{} {
	var p struct {
		URI    string `json:"uri"`
		Line   int    `json:"line"`
		Column int    `json:"column"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	req := &Request{
		JSONRPC: "2.0",
		Method:  "textDocument/definition",
		Params: map[string]interface{}{
			"textDocument": map[string]string{"uri": p.URI},
			"position": map[string]int{
				"line":      p.Line,
				"character": p.Column,
			},
		},
	}

	resp, err := m.sendRequest(req)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{"result": resp.Result}
}

func (m *Manager) handleReferences(params json.RawMessage) interface{} {
	var p struct {
		URI    string `json:"uri"`
		Line   int    `json:"line"`
		Column int    `json:"column"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	req := &Request{
		JSONRPC: "2.0",
		Method:  "textDocument/references",
		Params: map[string]interface{}{
			"textDocument": map[string]string{"uri": p.URI},
			"position": map[string]int{
				"line":      p.Line,
				"character": p.Column,
			},
			"context": map[string]bool{
				"includeDeclaration": true,
			},
		},
	}

	resp, err := m.sendRequest(req)
	if err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	return map[string]interface{}{"result": resp.Result}
}

func (m *Manager) Shutdown() error {
	m.cancel()
	if m.process != nil {
		return m.process.Process.Kill()
	}
	return nil
}
