package plugins

import (
	"encoding/json"
	"fmt"
	"sync"

	"github.com/rs/zerolog/log"
)

// Manager handles plugin loading and execution
type Manager struct {
	plugins map[string]*Plugin
	mu      sync.Mutex
}

type Plugin struct {
	ID          string   `json:"id"`
	Name        string   `json:"name"`
	Version     string   `json:"version"`
	Description string   `json:"description"`
	Author      string   `json:"author"`
	Permissions []string `json:"permissions"`
	Active      bool     `json:"active"`
}

func NewManager() *Manager {
	return &Manager{
		plugins: make(map[string]*Plugin),
	}
}

// Handle processes plugin commands from WebSocket
func (m *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling plugin command")

	switch command {
	case "list":
		return m.handleList(params)
	case "load":
		return m.handleLoad(params)
	case "unload":
		return m.handleUnload(params)
	case "info":
		return m.handleInfo(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown plugin command: %s", command),
		}
	}
}

func (m *Manager) handleList(params json.RawMessage) interface{} {
	m.mu.Lock()
	defer m.mu.Unlock()

	plugins := []Plugin{}
	for _, plugin := range m.plugins {
		plugins = append(plugins, *plugin)
	}

	return map[string]interface{}{
		"plugins": plugins,
		"total":   len(plugins),
	}
}

func (m *Manager) handleLoad(params json.RawMessage) interface{} {
	var p struct {
		Name        string   `json:"name"`
		Version     string   `json:"version"`
		Description string   `json:"description"`
		Author      string   `json:"author"`
		Permissions []string `json:"permissions"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	m.mu.Lock()
	defer m.mu.Unlock()

	pluginID := fmt.Sprintf("plugin-%s", p.Name)

	plugin := &Plugin{
		ID:          pluginID,
		Name:        p.Name,
		Version:     p.Version,
		Description: p.Description,
		Author:      p.Author,
		Permissions: p.Permissions,
		Active:      true,
	}

	m.plugins[pluginID] = plugin

	log.Info().Str("plugin", p.Name).Msg("Plugin loaded")

	return map[string]interface{}{
		"plugin":  plugin,
		"message": fmt.Sprintf("Plugin '%s' loaded successfully", p.Name),
	}
}

func (m *Manager) handleUnload(params json.RawMessage) interface{} {
	var p struct {
		ID string `json:"id"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	m.mu.Lock()
	defer m.mu.Unlock()

	plugin, ok := m.plugins[p.ID]
	if !ok {
		return map[string]interface{}{"error": "plugin not found"}
	}

	delete(m.plugins, p.ID)

	log.Info().Str("plugin", plugin.Name).Msg("Plugin unloaded")

	return map[string]interface{}{
		"id":      p.ID,
		"message": fmt.Sprintf("Plugin '%s' unloaded successfully", plugin.Name),
	}
}

func (m *Manager) handleInfo(params json.RawMessage) interface{} {
	var p struct {
		ID string `json:"id"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	m.mu.Lock()
	defer m.mu.Unlock()

	plugin, ok := m.plugins[p.ID]
	if !ok {
		return map[string]interface{}{"error": "plugin not found"}
	}

	return map[string]interface{}{
		"plugin": plugin,
	}
}

// Example plugin API context
type PluginContext struct {
	PluginID string
	Commands *CommandRegistry
	Editor   *EditorAPI
	UI       *UIAPI
}

type CommandRegistry struct{}

func (c *CommandRegistry) Register(id string, handler func()) error {
	log.Info().Str("commandId", id).Msg("Command registered")
	return nil
}

type EditorAPI struct{}

func (e *EditorAPI) GetActiveEditor() string {
	return "main.go"
}

type UIAPI struct{}

func (u *UIAPI) ShowNotification(msg string) {
	log.Info().Str("message", msg).Msg("Notification shown")
}
