package server

import (
	"encoding/json"
	"fmt"
	"net/http"
	"sync"
	"time"

	"github.com/gorilla/websocket"
	"github.com/rs/zerolog"
	"github.com/rs/zerolog/log"
	"github.com/yourorg/go-ide/internal/dap"
	"github.com/yourorg/go-ide/internal/files"
	"github.com/yourorg/go-ide/internal/git"
	"github.com/yourorg/go-ide/internal/lsp"
	"github.com/yourorg/go-ide/internal/plugins"
	"github.com/yourorg/go-ide/internal/terminal"
)

// StaticFS is a placeholder for embedded files
type StaticFS struct{}

type Config struct {
	Port        string
	Workspace   string
	StaticFiles StaticFS
	Verbose     bool
}

type Server struct {
	config   *Config
	lsp      *lsp.Manager
	dap      *dap.Manager
	git      *git.Manager
	files    *files.Manager
	terminal *terminal.Manager
	plugins  *plugins.Manager
	upgrader websocket.Upgrader
	clients  map[*websocket.Conn]bool
	mu       sync.Mutex
}

func New(cfg *Config) *Server {
	// Configure logging
	if cfg.Verbose {
		zerolog.SetGlobalLevel(zerolog.DebugLevel)
	} else {
		zerolog.SetGlobalLevel(zerolog.InfoLevel)
	}

	return &Server{
		config:  cfg,
		clients: make(map[*websocket.Conn]bool),
		upgrader: websocket.Upgrader{
			CheckOrigin: func(r *http.Request) bool {
				return true // Allow localhost connections
			},
		},
	}
}

func (s *Server) Start() error {
	var err error

	// Initialize LSP manager
	s.lsp, err = lsp.NewManager(s.config.Workspace)
	if err != nil {
		log.Warn().Err(err).Msg("LSP manager initialization failed (continuing without LSP)")
	} else {
		log.Info().Msg("LSP manager initialized")
	}

	// Initialize DAP manager
	s.dap = dap.NewManager()
	log.Info().Msg("DAP manager initialized")

	// Initialize Git manager
	s.git, err = git.NewManager(s.config.Workspace)
	if err != nil {
		log.Warn().Err(err).Msg("Git repository not found (continuing without Git)")
	} else {
		log.Info().Msg("Git manager initialized")
	}

	// Initialize File manager
	s.files, err = files.NewManager(s.config.Workspace)
	if err != nil {
		return fmt.Errorf("failed to start file manager: %w", err)
	}
	log.Info().Msg("File manager initialized")

	// Initialize Terminal manager
	s.terminal = terminal.NewManager(s.config.Workspace)
	log.Info().Msg("Terminal manager initialized")

	// Initialize Plugin manager
	s.plugins = plugins.NewManager()
	log.Info().Msg("Plugin manager initialized")

	// Setup HTTP routes
	mux := http.NewServeMux()

	// WebSocket endpoint
	mux.HandleFunc("/ws", s.handleWebSocket)

	// API endpoints
	mux.HandleFunc("/api/health", s.handleHealth)

	// Static files (frontend) - serve placeholder for now
	mux.HandleFunc("/", s.handlePlaceholder)

	// HTTP server
	srv := &http.Server{
		Addr:         ":" + s.config.Port,
		Handler:      mux,
		ReadTimeout:  15 * time.Second,
		WriteTimeout: 15 * time.Second,
	}

	log.Info().Str("addr", srv.Addr).Msg("Server starting")
	return srv.ListenAndServe()
}

func (s *Server) handlePlaceholder(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "text/html")
	fmt.Fprintf(w, `<!DOCTYPE html>
<html>
<head>
    <title>Go IDE Platform</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; }
        h1 { color: #00ADD8; }
        .status { background: #f0f9ff; border-left: 4px solid #00ADD8; padding: 15px; margin: 20px 0; }
        code { background: #f5f5f5; padding: 2px 6px; border-radius: 3px; }
    </style>
</head>
<body>
    <h1>🚀 Go IDE Platform</h1>
    <div class="status">
        <p><strong>Status:</strong> Backend Running</p>
        <p><strong>Workspace:</strong> %s</p>
        <p><strong>WebSocket:</strong> ws://localhost:%s/ws</p>
    </div>
    <h2>Available Services</h2>
    <ul>
        <li>LSP Integration (gopls) - Code completion, hover, definitions</li>
        <li>DAP Debugger (delve) - Breakpoints, stepping, variable inspection</li>
        <li>Git Integration - Status, diff, commit, branches</li>
        <li>File Management - Read, write, watch</li>
        <li>Terminal Emulation - Execute shell commands</li>
        <li>Plugin System - WASM and Go plugins</li>
    </ul>
    <h2>API Endpoints</h2>
    <ul>
        <li><code>GET /api/health</code> - Health check</li>
        <li><code>WS /ws</code> - WebSocket connection for IDE operations</li>
    </ul>
    <p><em>Note: Frontend assets not embedded. Build with <code>make build</code> to enable full UI.</em></p>
</body>
</html>`, s.config.Workspace, s.config.Port)
}

func (s *Server) handleHealth(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json")
	status := map[string]interface{}{
		"status":    "healthy",
		"workspace": s.config.Workspace,
		"services": map[string]bool{
			"lsp":      s.lsp != nil,
			"dap":      s.dap != nil,
			"git":      s.git != nil,
			"files":    s.files != nil,
			"terminal": s.terminal != nil,
			"plugins":  s.plugins != nil,
		},
	}
	json.NewEncoder(w).Encode(status)
}

func (s *Server) handleWebSocket(w http.ResponseWriter, r *http.Request) {
	conn, err := s.upgrader.Upgrade(w, r, nil)
	if err != nil {
		log.Error().Err(err).Msg("WebSocket upgrade failed")
		return
	}

	s.mu.Lock()
	s.clients[conn] = true
	s.mu.Unlock()

	defer func() {
		s.mu.Lock()
		delete(s.clients, conn)
		s.mu.Unlock()
		conn.Close()
	}()

	log.Info().Msg("Client connected via WebSocket")

	for {
		var msg Message
		if err := conn.ReadJSON(&msg); err != nil {
			if websocket.IsUnexpectedCloseError(err, websocket.CloseGoingAway, websocket.CloseAbnormalClosure) {
				log.Error().Err(err).Msg("WebSocket read error")
			}
			break
		}

		response := s.routeMessage(&msg)

		if err := conn.WriteJSON(response); err != nil {
			log.Error().Err(err).Msg("WebSocket write error")
			break
		}
	}

	log.Info().Msg("Client disconnected")
}

func (s *Server) routeMessage(msg *Message) interface{} {
	log.Debug().
		Str("id", msg.ID).
		Str("type", msg.Type).
		Str("command", msg.Command).
		Msg("Routing message")

	switch msg.Type {
	case "lsp":
		if s.lsp == nil {
			return ErrorResponse{ID: msg.ID, Error: "LSP not available"}
		}
		return s.lsp.Handle(msg.Command, msg.Params)

	case "dap":
		if s.dap == nil {
			return ErrorResponse{ID: msg.ID, Error: "DAP not available"}
		}
		return s.dap.Handle(msg.Command, msg.Params)

	case "git":
		if s.git == nil {
			return ErrorResponse{ID: msg.ID, Error: "Git not available"}
		}
		return s.git.Handle(msg.Command, msg.Params)

	case "file":
		if s.files == nil {
			return ErrorResponse{ID: msg.ID, Error: "File manager not available"}
		}
		return s.files.Handle(msg.Command, msg.Params)

	case "terminal":
		if s.terminal == nil {
			return ErrorResponse{ID: msg.ID, Error: "Terminal not available"}
		}
		return s.terminal.Handle(msg.Command, msg.Params)

	case "plugin":
		if s.plugins == nil {
			return ErrorResponse{ID: msg.ID, Error: "Plugin manager not available"}
		}
		return s.plugins.Handle(msg.Command, msg.Params)

	default:
		return ErrorResponse{
			ID:    msg.ID,
			Error: fmt.Sprintf("unknown message type: %s", msg.Type),
		}
	}
}

type Message struct {
	ID      string          `json:"id"`
	Type    string          `json:"type"`
	Command string          `json:"command"`
	Params  json.RawMessage `json:"params"`
}

type ErrorResponse struct {
	ID    string `json:"id"`
	Error string `json:"error"`
}
