package terminal

import (
	"encoding/json"
	"fmt"
	"os/exec"
	"sync"

	"github.com/rs/zerolog/log"
)

// Manager handles terminal sessions
type Manager struct {
	sessions map[string]*Session
	workDir  string
	mu       sync.Mutex
}

type Session struct {
	ID      string
	Command string
	Output  string
	Running bool
}

func NewManager(workDir string) *Manager {
	return &Manager{
		sessions: make(map[string]*Session),
		workDir:  workDir,
	}
}

// Handle processes terminal commands from WebSocket
func (m *Manager) Handle(command string, params json.RawMessage) interface{} {
	log.Debug().Str("command", command).Msg("Handling terminal command")

	switch command {
	case "create":
		return m.handleCreate(params)
	case "execute":
		return m.handleExecute(params)
	case "list":
		return m.handleList(params)
	case "close":
		return m.handleClose(params)
	default:
		return map[string]interface{}{
			"error": fmt.Sprintf("unknown terminal command: %s", command),
		}
	}
}

func (m *Manager) handleCreate(params json.RawMessage) interface{} {
	m.mu.Lock()
	defer m.mu.Unlock()

	sessionID := fmt.Sprintf("term-%d", len(m.sessions)+1)

	session := &Session{
		ID:      sessionID,
		Running: false,
	}

	m.sessions[sessionID] = session

	log.Info().Str("sessionId", sessionID).Msg("Terminal session created")

	return map[string]interface{}{
		"sessionId": sessionID,
		"message":   "Terminal session created",
	}
}

func (m *Manager) handleExecute(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
		Command   string `json:"command"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	m.mu.Lock()
	session, ok := m.sessions[p.SessionID]
	m.mu.Unlock()

	if !ok {
		return map[string]interface{}{"error": "session not found"}
	}

	session.Running = true
	session.Command = p.Command

	// Execute command
	cmd := exec.Command("sh", "-c", p.Command)
	cmd.Dir = m.workDir

	output, err := cmd.CombinedOutput()

	session.Running = false
	session.Output = string(output)

	if err != nil {
		log.Warn().Err(err).Str("command", p.Command).Msg("Command execution failed")
		return map[string]interface{}{
			"sessionId": p.SessionID,
			"output":    string(output),
			"error":     err.Error(),
			"exitCode":  cmd.ProcessState.ExitCode(),
		}
	}

	log.Info().Str("command", p.Command).Msg("Command executed successfully")

	return map[string]interface{}{
		"sessionId": p.SessionID,
		"output":    string(output),
		"exitCode":  0,
	}
}

func (m *Manager) handleList(params json.RawMessage) interface{} {
	m.mu.Lock()
	defer m.mu.Unlock()

	sessions := []map[string]interface{}{}
	for id, session := range m.sessions {
		sessions = append(sessions, map[string]interface{}{
			"id":      id,
			"command": session.Command,
			"running": session.Running,
		})
	}

	return map[string]interface{}{
		"sessions": sessions,
		"total":    len(sessions),
	}
}

func (m *Manager) handleClose(params json.RawMessage) interface{} {
	var p struct {
		SessionID string `json:"sessionId"`
	}
	if err := json.Unmarshal(params, &p); err != nil {
		return map[string]interface{}{"error": err.Error()}
	}

	m.mu.Lock()
	defer m.mu.Unlock()

	if _, ok := m.sessions[p.SessionID]; !ok {
		return map[string]interface{}{"error": "session not found"}
	}

	delete(m.sessions, p.SessionID)

	log.Info().Str("sessionId", p.SessionID).Msg("Terminal session closed")

	return map[string]interface{}{
		"sessionId": p.SessionID,
		"message":   "Terminal session closed",
	}
}
