package main

import (
	"fmt"
	"go/ast"
	"go/parser"
	"go/token"
)

// LinterPlugin provides basic Go code linting
type LinterPlugin struct {
	name    string
	version string
	issues  []Issue
}

// Issue represents a linting issue
type Issue struct {
	File     string `json:"file"`
	Line     int    `json:"line"`
	Column   int    `json:"column"`
	Severity string `json:"severity"` // "error", "warning", "info"
	Message  string `json:"message"`
}

// NewLinterPlugin creates a new linter plugin
func NewLinterPlugin() *LinterPlugin {
	return &LinterPlugin{
		name:    "go-linter",
		version: "1.0.0",
		issues:  make([]Issue, 0),
	}
}

// Name returns the plugin name
func (p *LinterPlugin) Name() string {
	return p.name
}

// Version returns the plugin version
func (p *LinterPlugin) Version() string {
	return p.version
}

// Description returns plugin description
func (p *LinterPlugin) Description() string {
	return "Basic Go code linter for common issues"
}

// Lint analyzes Go source code for issues
func (p *LinterPlugin) Lint(filename string, source []byte) ([]Issue, error) {
	p.issues = make([]Issue, 0)

	fset := token.NewFileSet()
	file, err := parser.ParseFile(fset, filename, source, parser.ParseComments)
	if err != nil {
		return nil, fmt.Errorf("parse error: %w", err)
	}

	// Simple linting rules
	ast.Inspect(file, func(n ast.Node) bool {
		switch x := n.(type) {
		case *ast.FuncDecl:
			// Check for functions without comments
			if x.Doc == nil && x.Name.IsExported() {
				pos := fset.Position(x.Pos())
				p.issues = append(p.issues, Issue{
					File:     filename,
					Line:     pos.Line,
					Column:   pos.Column,
					Severity: "warning",
					Message:  fmt.Sprintf("exported function %s should have comment", x.Name.Name),
				})
			}

		case *ast.GenDecl:
			// Check for exported types without comments
			if x.Tok == token.TYPE && x.Doc == nil {
				for _, spec := range x.Specs {
					if ts, ok := spec.(*ast.TypeSpec); ok && ts.Name.IsExported() {
						pos := fset.Position(ts.Pos())
						p.issues = append(p.issues, Issue{
							File:     filename,
							Line:     pos.Line,
							Column:   pos.Column,
							Severity: "warning",
							Message:  fmt.Sprintf("exported type %s should have comment", ts.Name.Name),
						})
					}
				}
			}
		}
		return true
	})

	return p.issues, nil
}

// Activate is called when the plugin is loaded
func (p *LinterPlugin) Activate() error {
	fmt.Println("Go linter plugin activated")
	return nil
}

// Deactivate is called when the plugin is unloaded
func (p *LinterPlugin) Deactivate() error {
	fmt.Println("Go linter plugin deactivated")
	return nil
}

// Example usage:
// plugin := NewLinterPlugin()
// issues, err := plugin.Lint("main.go", source)
// for _, issue := range issues {
//     fmt.Printf("%s:%d:%d: %s: %s\n", issue.File, issue.Line, issue.Column, issue.Severity, issue.Message)
// }
