package main

import (
	"blockchain-network/pkg/api"
	"blockchain-network/pkg/blockchain"
	"blockchain-network/pkg/consensus"
	"blockchain-network/pkg/mempool"
	"blockchain-network/pkg/p2p"
	"blockchain-network/pkg/wallet"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"
)

var (
	apiPort  int
	p2pPort  int
	dataDir  string
	bootstrap string
)

func main() {
	rootCmd := &cobra.Command{
		Use:   "blockchain",
		Short: "Blockchain Network Node",
		Run:   runNode,
	}

	rootCmd.Flags().IntVar(&apiPort, "api-port", 8545, "API server port")
	rootCmd.Flags().IntVar(&p2pPort, "p2p-port", 30303, "P2P network port")
	rootCmd.Flags().StringVar(&dataDir, "data-dir", "./data", "Data directory")
	rootCmd.Flags().StringVar(&bootstrap, "bootstrap", "", "Bootstrap peer address")

	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func runNode(cmd *cobra.Command, args []string) {
	fmt.Println("Starting Blockchain Node...")

	// Create blockchain
	bc := blockchain.NewBlockchain()
	fmt.Printf("Genesis block created: %x\n", bc.GetLatestBlock().Hash())

	// Create mempool
	mp := mempool.NewMempool()

	// Create wallet
	w, err := wallet.NewWallet()
	if err != nil {
		fmt.Printf("Failed to create wallet: %v\n", err)
		return
	}
	fmt.Printf("Wallet created: %s\n", w.Address)

	// Create miner
	miner := consensus.NewMiner(bc, mp, w.Address)

	// Create P2P network
	network := p2p.NewNetwork(p2pPort, bc, mp)
	if err := network.Start(); err != nil {
		fmt.Printf("Failed to start P2P network: %v\n", err)
		return
	}

	// Connect to bootstrap peer if provided
	if bootstrap != "" {
		fmt.Printf("Connecting to bootstrap peer: %s\n", bootstrap)
		if err := network.ConnectToPeer(bootstrap); err != nil {
			fmt.Printf("Failed to connect to bootstrap peer: %v\n", err)
		}
	}

	// Start API server
	apiServer := api.NewAPI(bc, mp, w, miner)
	go func() {
		if err := apiServer.Start(apiPort); err != nil {
			fmt.Printf("API server error: %v\n", err)
		}
	}()

	// Start mining
	fmt.Println("Starting miner...")
	miner.Start()

	// Wait for interrupt signal
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)
	<-sigChan

	fmt.Println("\nShutting down...")
	miner.Stop()
}
