package api

import (
	"blockchain-network/pkg/blockchain"
	"blockchain-network/pkg/consensus"
	"blockchain-network/pkg/mempool"
	"blockchain-network/pkg/wallet"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
)

// API provides REST endpoints
type API struct {
	blockchain *blockchain.Blockchain
	mempool    *mempool.Mempool
	wallet     *wallet.Wallet
	miner      *consensus.Miner
}

// NewAPI creates a new API instance
func NewAPI(bc *blockchain.Blockchain, mp *mempool.Mempool, w *wallet.Wallet, m *consensus.Miner) *API {
	return &API{
		blockchain: bc,
		mempool:    mp,
		wallet:     w,
		miner:      m,
	}
}

// Start begins API server
func (api *API) Start(port int) error {
	r := chi.NewRouter()

	// Middleware
	r.Use(middleware.Logger)
	r.Use(middleware.Recoverer)

	// Blockchain endpoints
	r.Get("/blocks/{height}", api.GetBlock)
	r.Get("/blocks/latest", api.GetLatestBlock)
	r.Get("/chain/info", api.GetChainInfo)

	// Transaction endpoints
	r.Post("/transactions", api.SendTransaction)
	r.Get("/transactions/{hash}", api.GetTransaction)
	r.Get("/mempool", api.GetMempool)

	// Wallet endpoints
	r.Get("/balance/{address}", api.GetBalance)
	r.Get("/wallet/address", api.GetWalletAddress)

	// Mining endpoints
	r.Post("/mining/start", api.StartMining)
	r.Post("/mining/stop", api.StopMining)
	r.Get("/mining/status", api.GetMiningStatus)

	fmt.Printf("API server listening on port %d\n", port)
	return http.ListenAndServe(fmt.Sprintf(":%d", port), r)
}

// GetBlock returns block by height
func (api *API) GetBlock(w http.ResponseWriter, r *http.Request) {
	heightStr := chi.URLParam(r, "height")
	height, err := strconv.ParseUint(heightStr, 10, 64)
	if err != nil {
		http.Error(w, "invalid height", http.StatusBadRequest)
		return
	}

	block, err := api.blockchain.GetBlock(height)
	if err != nil {
		http.Error(w, "block not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(block)
}

// GetLatestBlock returns the newest block
func (api *API) GetLatestBlock(w http.ResponseWriter, r *http.Request) {
	block := api.blockchain.GetLatestBlock()
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(block)
}

// GetChainInfo returns blockchain statistics
func (api *API) GetChainInfo(w http.ResponseWriter, r *http.Request) {
	latestBlock := api.blockchain.GetLatestBlock()
	info := map[string]interface{}{
		"height":       api.blockchain.GetHeight(),
		"latest_hash":  hex.EncodeToString(latestBlock.Hash()),
		"pending_txs":  api.mempool.Size(),
		"difficulty":   latestBlock.Header.Difficulty,
		"timestamp":    latestBlock.Header.Timestamp,
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(info)
}

// SendTransaction handles transaction submission
func (api *API) SendTransaction(w http.ResponseWriter, r *http.Request) {
	var tx blockchain.Transaction
	if err := json.NewDecoder(r.Body).Decode(&tx); err != nil {
		http.Error(w, "invalid transaction", http.StatusBadRequest)
		return
	}

	if err := api.mempool.AddTransaction(&tx); err != nil {
		http.Error(w, err.Error(), http.StatusBadRequest)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(map[string]string{
		"status": "success",
		"hash":   tx.HashString(),
	})
}

// GetTransaction retrieves transaction by hash
func (api *API) GetTransaction(w http.ResponseWriter, r *http.Request) {
	hash := chi.URLParam(r, "hash")

	tx, exists := api.mempool.GetTransaction(hash)
	if !exists {
		http.Error(w, "transaction not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(tx)
}

// GetMempool returns pending transactions
func (api *API) GetMempool(w http.ResponseWriter, r *http.Request) {
	txs := api.mempool.GetTransactions(0)
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(txs)
}

// GetBalance returns account balance (simplified)
func (api *API) GetBalance(w http.ResponseWriter, r *http.Request) {
	address := chi.URLParam(r, "address")

	// In a real implementation, we'd scan the blockchain
	// For now, return a placeholder
	balance := map[string]interface{}{
		"address": address,
		"balance": 0,
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(balance)
}

// GetWalletAddress returns wallet address
func (api *API) GetWalletAddress(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(map[string]string{
		"address": api.wallet.Address,
	})
}

// StartMining starts mining
func (api *API) StartMining(w http.ResponseWriter, r *http.Request) {
	api.miner.Start()
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(map[string]string{
		"status": "mining started",
	})
}

// StopMining stops mining
func (api *API) StopMining(w http.ResponseWriter, r *http.Request) {
	api.miner.Stop()
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(map[string]string{
		"status": "mining stopped",
	})
}

// GetMiningStatus returns mining status
func (api *API) GetMiningStatus(w http.ResponseWriter, r *http.Request) {
	status := map[string]interface{}{
		"mining": api.miner.IsMining(),
	}
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(status)
}
