package blockchain

import (
	"bytes"
	"crypto/sha256"
	"encoding/binary"
	"time"
)

// Block represents a block in the blockchain
type Block struct {
	Header       *BlockHeader   `json:"header"`
	Transactions []*Transaction `json:"transactions"`
}

// BlockHeader contains block metadata
type BlockHeader struct {
	Version       uint32 `json:"version"`
	PrevBlockHash []byte `json:"prev_block_hash"`
	MerkleRoot    []byte `json:"merkle_root"`
	Timestamp     int64  `json:"timestamp"`
	Difficulty    uint32 `json:"difficulty"`
	Nonce         uint64 `json:"nonce"`
	Height        uint64 `json:"height"`
}

// Hash calculates the block hash
func (b *Block) Hash() []byte {
	headerBytes := b.Header.Serialize()
	hash := sha256.Sum256(headerBytes)
	return hash[:]
}

// Serialize converts block header to bytes
func (bh *BlockHeader) Serialize() []byte {
	var buf bytes.Buffer

	binary.Write(&buf, binary.BigEndian, bh.Version)
	buf.Write(bh.PrevBlockHash)
	buf.Write(bh.MerkleRoot)
	binary.Write(&buf, binary.BigEndian, bh.Timestamp)
	binary.Write(&buf, binary.BigEndian, bh.Difficulty)
	binary.Write(&buf, binary.BigEndian, bh.Nonce)
	binary.Write(&buf, binary.BigEndian, bh.Height)

	return buf.Bytes()
}

// NewGenesisBlock creates the first block in the chain
func NewGenesisBlock() *Block {
	tx := &Transaction{
		From:  "0x0000000000000000000000000000000000000000",
		To:    "0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb", // Genesis address
		Value: 1000000000,                                  // 1 billion tokens
		Data:  []byte("Genesis Block"),
	}
	tx.Hash = tx.CalculateHash()

	merkleRoot := CalculateMerkleRoot([]*Transaction{tx})

	header := &BlockHeader{
		Version:       1,
		PrevBlockHash: make([]byte, 32), // All zeros
		MerkleRoot:    merkleRoot,
		Timestamp:     time.Now().Unix(),
		Difficulty:    20, // Easy difficulty for testing
		Nonce:         0,
		Height:        0,
	}

	return &Block{
		Header:       header,
		Transactions: []*Transaction{tx},
	}
}
