package blockchain

import (
	"bytes"
	"crypto/sha256"
	"errors"
	"fmt"
	"math/big"
	"sync"
)

// Blockchain manages the chain
type Blockchain struct {
	blocks []*Block
	mu     sync.RWMutex
}

// NewBlockchain creates a new blockchain with genesis block
func NewBlockchain() *Blockchain {
	return &Blockchain{
		blocks: []*Block{NewGenesisBlock()},
	}
}

// AddBlock adds a new block to the chain
func (bc *Blockchain) AddBlock(block *Block) error {
	bc.mu.Lock()
	defer bc.mu.Unlock()

	// Validate block
	if err := bc.validateBlock(block); err != nil {
		return err
	}

	bc.blocks = append(bc.blocks, block)
	return nil
}

// validateBlock checks if block is valid
func (bc *Blockchain) validateBlock(block *Block) error {
	lastBlock := bc.blocks[len(bc.blocks)-1]

	// Check previous hash
	if !bytes.Equal(block.Header.PrevBlockHash, lastBlock.Hash()) {
		return errors.New("invalid previous block hash")
	}

	// Check height
	if block.Header.Height != lastBlock.Header.Height+1 {
		return errors.New("invalid block height")
	}

	// Verify merkle root
	calculatedRoot := CalculateMerkleRoot(block.Transactions)
	if !bytes.Equal(block.Header.MerkleRoot, calculatedRoot) {
		return errors.New("invalid merkle root")
	}

	// Verify PoW
	if !verifyProofOfWork(block) {
		return errors.New("invalid proof of work")
	}

	return nil
}

// verifyProofOfWork checks if proof of work is valid
func verifyProofOfWork(block *Block) bool {
	var hashInt big.Int

	hash := sha256.Sum256(block.Header.Serialize())
	hashInt.SetBytes(hash[:])

	target := big.NewInt(1)
	target.Lsh(target, uint(256-block.Header.Difficulty))

	return hashInt.Cmp(target) == -1
}

// GetBlock retrieves block by height
func (bc *Blockchain) GetBlock(height uint64) (*Block, error) {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	if height >= uint64(len(bc.blocks)) {
		return nil, errors.New("block not found")
	}

	return bc.blocks[height], nil
}

// GetLatestBlock returns the newest block
func (bc *Blockchain) GetLatestBlock() *Block {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	return bc.blocks[len(bc.blocks)-1]
}

// GetHeight returns current chain height
func (bc *Blockchain) GetHeight() uint64 {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	return uint64(len(bc.blocks) - 1)
}

// GetBlocks returns all blocks
func (bc *Blockchain) GetBlocks() []*Block {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	blocks := make([]*Block, len(bc.blocks))
	copy(blocks, bc.blocks)
	return blocks
}

// String returns blockchain info
func (bc *Blockchain) String() string {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	return fmt.Sprintf("Blockchain{height=%d, blocks=%d}",
		bc.GetHeight(), len(bc.blocks))
}
