package blockchain

import "crypto/sha256"

// CalculateMerkleRoot builds merkle tree from transactions
func CalculateMerkleRoot(transactions []*Transaction) []byte {
	if len(transactions) == 0 {
		return make([]byte, 32)
	}

	var nodes [][]byte
	for _, tx := range transactions {
		nodes = append(nodes, tx.CalculateHash())
	}

	// Build tree bottom-up
	for len(nodes) > 1 {
		var newLevel [][]byte

		for i := 0; i < len(nodes); i += 2 {
			if i+1 < len(nodes) {
				newLevel = append(newLevel, HashPair(nodes[i], nodes[i+1]))
			} else {
				// Odd number of nodes, duplicate last one
				newLevel = append(newLevel, HashPair(nodes[i], nodes[i]))
			}
		}

		nodes = newLevel
	}

	return nodes[0]
}

// HashPair hashes two nodes together
func HashPair(left, right []byte) []byte {
	combined := append(left, right...)
	hash := sha256.Sum256(combined)
	return hash[:]
}

// GenerateMerkleProof generates proof for transaction inclusion
func GenerateMerkleProof(transactions []*Transaction, txIndex int) [][]byte {
	var proof [][]byte
	var nodes [][]byte

	for _, tx := range transactions {
		nodes = append(nodes, tx.CalculateHash())
	}

	index := txIndex

	for len(nodes) > 1 {
		var newLevel [][]byte

		for i := 0; i < len(nodes); i += 2 {
			if i+1 < len(nodes) {
				// Add sibling to proof
				if i == index || i+1 == index {
					if i == index {
						proof = append(proof, nodes[i+1])
					} else {
						proof = append(proof, nodes[i])
					}
				}
				newLevel = append(newLevel, HashPair(nodes[i], nodes[i+1]))
			} else {
				newLevel = append(newLevel, HashPair(nodes[i], nodes[i]))
			}
		}

		index = index / 2
		nodes = newLevel
	}

	return proof
}
