package blockchain

import (
	"crypto/ecdsa"
	"crypto/rand"
	"crypto/sha256"
	"encoding/hex"
	"fmt"
	"math/big"
)

// Transaction represents a blockchain transaction
type Transaction struct {
	From      string `json:"from"`       // Address (hex string)
	To        string `json:"to"`         // Address (hex string)
	Value     uint64 `json:"value"`      // Amount to transfer
	Data      []byte `json:"data"`       // Contract bytecode or call data
	Nonce     uint64 `json:"nonce"`      // Transaction counter
	GasPrice  uint64 `json:"gas_price"`  // Price per gas unit
	GasLimit  uint64 `json:"gas_limit"`  // Maximum gas allowed
	Signature []byte `json:"signature"`  // ECDSA signature
	Hash      []byte `json:"hash"`       // Transaction hash
}

// CalculateHash computes transaction hash
func (tx *Transaction) CalculateHash() []byte {
	data := fmt.Sprintf("%s%s%d%s%d%d%d",
		tx.From, tx.To, tx.Value, string(tx.Data),
		tx.Nonce, tx.GasPrice, tx.GasLimit)
	hash := sha256.Sum256([]byte(data))
	return hash[:]
}

// Sign signs the transaction with private key
func (tx *Transaction) Sign(privKey *ecdsa.PrivateKey) error {
	tx.Hash = tx.CalculateHash()

	r, s, err := ecdsa.Sign(rand.Reader, privKey, tx.Hash)
	if err != nil {
		return err
	}

	// Combine r and s into signature (pad to 32 bytes each)
	rBytes := r.Bytes()
	sBytes := s.Bytes()

	signature := make([]byte, 64)
	copy(signature[32-len(rBytes):32], rBytes)
	copy(signature[64-len(sBytes):64], sBytes)

	tx.Signature = signature

	return nil
}

// Verify verifies transaction signature
func (tx *Transaction) Verify(pubKey *ecdsa.PublicKey) bool {
	if len(tx.Signature) < 64 {
		return false
	}

	// Extract r and s from signature
	r := new(big.Int).SetBytes(tx.Signature[:32])
	s := new(big.Int).SetBytes(tx.Signature[32:64])

	// Recalculate hash to verify
	hash := tx.CalculateHash()

	// Verify signature
	return ecdsa.Verify(pubKey, hash, r, s)
}

// HashString returns hex encoded hash
func (tx *Transaction) HashString() string {
	return hex.EncodeToString(tx.Hash)
}
