package consensus

import (
	"blockchain-network/pkg/blockchain"
	"blockchain-network/pkg/mempool"
	"fmt"
	"sync"
	"time"
)

// Miner handles block mining
type Miner struct {
	blockchain    *blockchain.Blockchain
	mempool       *mempool.Mempool
	rewardAddress string
	mining        bool
	mu            sync.Mutex
}

// NewMiner creates a new miner
func NewMiner(bc *blockchain.Blockchain, mp *mempool.Mempool, rewardAddr string) *Miner {
	return &Miner{
		blockchain:    bc,
		mempool:       mp,
		rewardAddress: rewardAddr,
		mining:        false,
	}
}

// Start begins mining process
func (m *Miner) Start() {
	m.mu.Lock()
	if m.mining {
		m.mu.Unlock()
		return
	}
	m.mining = true
	m.mu.Unlock()

	go m.mineLoop()
}

// Stop halts mining
func (m *Miner) Stop() {
	m.mu.Lock()
	m.mining = false
	m.mu.Unlock()
}

// IsMining returns mining status
func (m *Miner) IsMining() bool {
	m.mu.Lock()
	defer m.mu.Unlock()
	return m.mining
}

// mineLoop continuously mines new blocks
func (m *Miner) mineLoop() {
	for {
		m.mu.Lock()
		if !m.mining {
			m.mu.Unlock()
			return
		}
		m.mu.Unlock()

		// Build new block
		block := m.buildBlock()

		// Mine block (find valid nonce)
		pow := NewProofOfWork(block)
		nonce, hash := pow.Mine()

		block.Header.Nonce = nonce

		// Add block to blockchain
		if err := m.blockchain.AddBlock(block); err != nil {
			fmt.Printf("Failed to add block: %v\n", err)
			continue
		}

		fmt.Printf("Block %d added to blockchain: %x\n",
			block.Header.Height, hash)

		// Remove mined transactions from mempool
		m.mempool.RemoveTransactions(block.Transactions)
	}
}

// buildBlock creates a new block from mempool transactions
func (m *Miner) buildBlock() *blockchain.Block {
	lastBlock := m.blockchain.GetLatestBlock()

	// Get transactions from mempool
	txs := m.mempool.GetTransactions(100) // Max 100 txs per block

	// Add coinbase transaction (mining reward)
	coinbase := &blockchain.Transaction{
		From:  "0x0000000000000000000000000000000000000000",
		To:    m.rewardAddress,
		Value: 50, // 50 tokens reward
		Data:  []byte("Mining Reward"),
		Nonce: 0,
	}
	coinbase.Hash = coinbase.CalculateHash()

	allTxs := append([]*blockchain.Transaction{coinbase}, txs...)

	header := &blockchain.BlockHeader{
		Version:       1,
		PrevBlockHash: lastBlock.Hash(),
		MerkleRoot:    blockchain.CalculateMerkleRoot(allTxs),
		Timestamp:     time.Now().Unix(),
		Difficulty:    m.calculateDifficulty(),
		Nonce:         0,
		Height:        lastBlock.Header.Height + 1,
	}

	return &blockchain.Block{
		Header:       header,
		Transactions: allTxs,
	}
}

// calculateDifficulty adjusts mining difficulty
func (m *Miner) calculateDifficulty() uint32 {
	// Simple difficulty adjustment
	// Target: 10 second block time
	// Adjust every 10 blocks

	height := m.blockchain.GetHeight()
	if height < 10 || height%10 != 0 {
		return m.blockchain.GetLatestBlock().Header.Difficulty
	}

	// Get block from 10 blocks ago
	oldBlock, _ := m.blockchain.GetBlock(height - 10)
	lastBlock := m.blockchain.GetLatestBlock()

	// Calculate actual time for last 10 blocks
	actualTime := lastBlock.Header.Timestamp - oldBlock.Header.Timestamp
	targetTime := int64(10 * 10) // 10 blocks * 10 seconds

	currentDifficulty := lastBlock.Header.Difficulty

	// Adjust difficulty
	if actualTime < targetTime/2 {
		// Mining too fast, increase difficulty
		return currentDifficulty + 1
	} else if actualTime > targetTime*2 {
		// Mining too slow, decrease difficulty
		if currentDifficulty > 1 {
			return currentDifficulty - 1
		}
	}

	return currentDifficulty
}
