package consensus

import (
	"blockchain-network/pkg/blockchain"
	"crypto/sha256"
	"fmt"
	"math/big"
	"time"
)

// ProofOfWork represents PoW consensus
type ProofOfWork struct {
	block  *blockchain.Block
	target *big.Int
}

// NewProofOfWork creates a new PoW instance
func NewProofOfWork(block *blockchain.Block) *ProofOfWork {
	target := big.NewInt(1)
	target.Lsh(target, uint(256-block.Header.Difficulty))

	return &ProofOfWork{
		block:  block,
		target: target,
	}
}

// Mine performs proof of work mining
func (pow *ProofOfWork) Mine() (uint64, []byte) {
	var hashInt big.Int
	var hash [32]byte
	nonce := uint64(0)

	fmt.Printf("Mining block at height %d with difficulty %d...\n",
		pow.block.Header.Height, pow.block.Header.Difficulty)

	startTime := time.Now()

	for {
		// Calculate hash with current nonce
		pow.block.Header.Nonce = nonce
		hash = sha256.Sum256(pow.block.Header.Serialize())
		hashInt.SetBytes(hash[:])

		// Check if hash meets difficulty target
		if hashInt.Cmp(pow.target) == -1 {
			duration := time.Since(startTime)
			fmt.Printf("Block mined! Nonce: %d, Hash: %x, Time: %s\n",
				nonce, hash, duration)
			break
		}

		nonce++

		// Print progress every 100k hashes
		if nonce%100000 == 0 {
			fmt.Printf("Tried %d hashes...\n", nonce)
		}
	}

	return nonce, hash[:]
}

// Verify checks if proof of work is valid
func VerifyProofOfWork(block *blockchain.Block) bool {
	var hashInt big.Int

	hash := sha256.Sum256(block.Header.Serialize())
	hashInt.SetBytes(hash[:])

	target := big.NewInt(1)
	target.Lsh(target, uint(256-block.Header.Difficulty))

	return hashInt.Cmp(target) == -1
}
