package mempool

import (
	"blockchain-network/pkg/blockchain"
	"errors"
	"sort"
	"sync"
)

// Mempool manages pending transactions
type Mempool struct {
	transactions map[string]*blockchain.Transaction // Hash -> Transaction
	mu           sync.RWMutex
}

// NewMempool creates a new transaction pool
func NewMempool() *Mempool {
	return &Mempool{
		transactions: make(map[string]*blockchain.Transaction),
	}
}

// AddTransaction adds transaction to pool
func (mp *Mempool) AddTransaction(tx *blockchain.Transaction) error {
	mp.mu.Lock()
	defer mp.mu.Unlock()

	// Check if already in pool
	txHash := tx.HashString()
	if _, exists := mp.transactions[txHash]; exists {
		return errors.New("transaction already in pool")
	}

	mp.transactions[txHash] = tx
	return nil
}

// GetTransactions returns transactions sorted by gas price
func (mp *Mempool) GetTransactions(limit int) []*blockchain.Transaction {
	mp.mu.RLock()
	defer mp.mu.RUnlock()

	txs := make([]*blockchain.Transaction, 0, len(mp.transactions))
	for _, tx := range mp.transactions {
		txs = append(txs, tx)
	}

	// Sort by gas price (descending)
	sort.Slice(txs, func(i, j int) bool {
		return txs[i].GasPrice > txs[j].GasPrice
	})

	if limit > 0 && len(txs) > limit {
		txs = txs[:limit]
	}

	return txs
}

// RemoveTransactions removes transactions from pool
func (mp *Mempool) RemoveTransactions(txs []*blockchain.Transaction) {
	mp.mu.Lock()
	defer mp.mu.Unlock()

	for _, tx := range txs {
		txHash := tx.HashString()
		delete(mp.transactions, txHash)
	}
}

// Size returns number of pending transactions
func (mp *Mempool) Size() int {
	mp.mu.RLock()
	defer mp.mu.RUnlock()

	return len(mp.transactions)
}

// GetTransaction retrieves transaction by hash
func (mp *Mempool) GetTransaction(hash string) (*blockchain.Transaction, bool) {
	mp.mu.RLock()
	defer mp.mu.RUnlock()

	tx, exists := mp.transactions[hash]
	return tx, exists
}
