package storage

import (
	"blockchain-network/pkg/blockchain"
	"encoding/json"
	"fmt"

	"github.com/dgraph-io/badger/v3"
)

// Storage manages blockchain persistence
type Storage struct {
	db *badger.DB
}

// NewStorage creates a new storage instance
func NewStorage(path string) (*Storage, error) {
	opts := badger.DefaultOptions(path)
	opts.Logger = nil // Disable logs

	db, err := badger.Open(opts)
	if err != nil {
		return nil, err
	}

	return &Storage{db: db}, nil
}

// SaveBlock persists block to disk
func (s *Storage) SaveBlock(block *blockchain.Block) error {
	return s.db.Update(func(txn *badger.Txn) error {
		// Save by height
		heightKey := []byte(fmt.Sprintf("block:%d", block.Header.Height))
		blockData, err := json.Marshal(block)
		if err != nil {
			return err
		}

		if err := txn.Set(heightKey, blockData); err != nil {
			return err
		}

		// Save hash -> height mapping
		hashKey := []byte(fmt.Sprintf("hash:%x", block.Hash()))
		heightData := []byte(fmt.Sprintf("%d", block.Header.Height))

		return txn.Set(hashKey, heightData)
	})
}

// GetBlock retrieves block by height
func (s *Storage) GetBlock(height uint64) (*blockchain.Block, error) {
	var block blockchain.Block

	err := s.db.View(func(txn *badger.Txn) error {
		key := []byte(fmt.Sprintf("block:%d", height))
		item, err := txn.Get(key)
		if err != nil {
			return err
		}

		return item.Value(func(val []byte) error {
			return json.Unmarshal(val, &block)
		})
	})

	return &block, err
}

// Close closes database
func (s *Storage) Close() error {
	return s.db.Close()
}
