package vm

import (
	"encoding/binary"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestVMPushPop(t *testing.T) {
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 5,
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 10,
		byte(POP),
		byte(RETURN),
	}

	vm := NewVM(code, 1000)
	result, err := vm.Execute()
	require.NoError(t, err)
	assert.Equal(t, "5", string(result))
}

func TestVMAdd(t *testing.T) {
	// PUSH 5, PUSH 10, ADD, RETURN
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 5,
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 10,
		byte(ADD),
		byte(RETURN),
	}

	vm := NewVM(code, 1000)
	result, err := vm.Execute()
	require.NoError(t, err)
	assert.Equal(t, "15", string(result))
}

func TestVMMultiply(t *testing.T) {
	// (5 + 10) * 3 = 45
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 5,
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 10,
		byte(ADD),
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 3,
		byte(MUL),
		byte(RETURN),
	}

	vm := NewVM(code, 1000)
	result, err := vm.Execute()
	require.NoError(t, err)
	assert.Equal(t, "45", string(result))
}

func TestVMStorage(t *testing.T) {
	// Store 42 at key 0, then load it
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 0,  // key
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 42, // value
		byte(SSTORE),
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 0, // key
		byte(SLOAD),
		byte(RETURN),
	}

	vm := NewVM(code, 1000)
	result, err := vm.Execute()
	require.NoError(t, err)
	assert.Equal(t, "42", string(result))
}

func TestVMComparison(t *testing.T) {
	// 10 > 5 should return 1
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 10,
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 5,
		byte(GT),
		byte(RETURN),
	}

	vm := NewVM(code, 1000)
	result, err := vm.Execute()
	require.NoError(t, err)
	assert.Equal(t, "1", string(result))
}

func TestVMOutOfGas(t *testing.T) {
	code := []byte{
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 5,
		byte(PUSH), 0, 0, 0, 0, 0, 0, 0, 10,
		byte(ADD),
		byte(RETURN),
	}

	vm := NewVM(code, 5) // Very low gas limit
	_, err := vm.Execute()
	require.Error(t, err)
	assert.Contains(t, err.Error(), "out of gas")
}

func makePushInstruction(value uint64) []byte {
	buf := make([]byte, 9)
	buf[0] = byte(PUSH)
	binary.BigEndian.PutUint64(buf[1:], value)
	return buf
}
