package wallet

import (
	"blockchain-network/pkg/blockchain"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/sha256"
	"encoding/hex"
)

// Wallet manages cryptographic keys
type Wallet struct {
	PrivateKey *ecdsa.PrivateKey
	PublicKey  *ecdsa.PublicKey
	Address    string
}

// NewWallet creates a new wallet
func NewWallet() (*Wallet, error) {
	privKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	if err != nil {
		return nil, err
	}

	pubKey := &privKey.PublicKey
	address := PublicKeyToAddress(pubKey)

	return &Wallet{
		PrivateKey: privKey,
		PublicKey:  pubKey,
		Address:    address,
	}, nil
}

// PublicKeyToAddress converts public key to address
func PublicKeyToAddress(pubKey *ecdsa.PublicKey) string {
	pubBytes := elliptic.Marshal(pubKey.Curve, pubKey.X, pubKey.Y)
	hash := sha256.Sum256(pubBytes)

	// Take last 20 bytes
	address := hash[len(hash)-20:]
	return "0x" + hex.EncodeToString(address)
}

// CreateTransaction creates and signs a transaction
func (w *Wallet) CreateTransaction(to string, value uint64, data []byte, nonce uint64) (*blockchain.Transaction, error) {
	tx := &blockchain.Transaction{
		From:     w.Address,
		To:       to,
		Value:    value,
		Data:     data,
		Nonce:    nonce,
		GasPrice: 1,
		GasLimit: 100000,
	}

	if err := tx.Sign(w.PrivateKey); err != nil {
		return nil, err
	}

	return tx, nil
}
