package main

import (
	"flag"
	"fmt"
	"os"
	"strconv"

	"taskmaster/internal/cli"
	"taskmaster/internal/models"
	"taskmaster/internal/storage"
)

const version = "1.0.0"

func main() {
	if len(os.Args) < 2 {
		printUsage()
		os.Exit(1)
	}

	// Initialize storage
	store, err := storage.NewSQLiteStorage("tasks.db")
	if err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
	defer store.Close()

	command := os.Args[1]

	switch command {
	case "add":
		handleAdd(store, os.Args[2:])
	case "list":
		handleList(store, os.Args[2:])
	case "complete":
		handleComplete(store, os.Args[2:])
	case "delete":
		handleDelete(store, os.Args[2:])
	case "version":
		fmt.Printf("taskmaster v%s\n", version)
	case "help":
		printUsage()
	default:
		fmt.Fprintf(os.Stderr, "Unknown command: %s\n\n", command)
		printUsage()
		os.Exit(1)
	}
}

func handleAdd(store storage.Storage, args []string) {
	fs := flag.NewFlagSet("add", flag.ExitOnError)
	description := fs.String("desc", "", "Task description")
	priority := fs.String("priority", "medium", "Priority (low|medium|high|critical)")
	fs.Parse(args)

	if fs.NArg() < 1 {
		fmt.Fprintln(os.Stderr, "Error: task title required")
		fmt.Fprintln(os.Stderr, "Usage: taskmaster add [flags] <title>")
		os.Exit(1)
	}

	title := fs.Arg(0)

	var p models.Priority
	switch *priority {
	case "low":
		p = models.PriorityLow
	case "medium":
		p = models.PriorityMedium
	case "high":
		p = models.PriorityHigh
	case "critical":
		p = models.PriorityCritical
	default:
		fmt.Fprintf(os.Stderr, "Invalid priority: %s\n", *priority)
		os.Exit(1)
	}

	if err := cli.AddTask(store, title, *description, p); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
}

func handleList(store storage.Storage, args []string) {
	fs := flag.NewFlagSet("list", flag.ExitOnError)
	status := fs.String("status", "all", "Filter by status (all|pending|completed)")
	priority := fs.String("priority", "", "Filter by priority (low|medium|high|critical)")
	fs.Parse(args)

	filter := &storage.TaskFilter{
		Status: *status,
	}

	if *priority != "" {
		var p models.Priority
		switch *priority {
		case "low":
			p = models.PriorityLow
		case "medium":
			p = models.PriorityMedium
		case "high":
			p = models.PriorityHigh
		case "critical":
			p = models.PriorityCritical
		default:
			fmt.Fprintf(os.Stderr, "Invalid priority: %s\n", *priority)
			os.Exit(1)
		}
		filter.Priority = &p
	}

	if err := cli.ListTasks(store, filter); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
}

func handleComplete(store storage.Storage, args []string) {
	if len(args) < 1 {
		fmt.Fprintln(os.Stderr, "Error: task ID required")
		fmt.Fprintln(os.Stderr, "Usage: taskmaster complete <id>")
		os.Exit(1)
	}

	id, err := strconv.Atoi(args[0])
	if err != nil {
		fmt.Fprintf(os.Stderr, "Invalid task ID: %s\n", args[0])
		os.Exit(1)
	}

	if err := cli.CompleteTask(store, id); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
}

func handleDelete(store storage.Storage, args []string) {
	if len(args) < 1 {
		fmt.Fprintln(os.Stderr, "Error: task ID required")
		fmt.Fprintln(os.Stderr, "Usage: taskmaster delete <id>")
		os.Exit(1)
	}

	id, err := strconv.Atoi(args[0])
	if err != nil {
		fmt.Fprintf(os.Stderr, "Invalid task ID: %s\n", args[0])
		os.Exit(1)
	}

	if err := cli.DeleteTask(store, id); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
}

func printUsage() {
	fmt.Println("TaskMaster - A simple CLI task manager")
	fmt.Println()
	fmt.Println("Usage:")
	fmt.Println("  taskmaster <command> [flags] [arguments]")
	fmt.Println()
	fmt.Println("Commands:")
	fmt.Println("  add        Add a new task")
	fmt.Println("  list       List all tasks")
	fmt.Println("  complete   Mark a task as complete")
	fmt.Println("  delete     Delete a task")
	fmt.Println("  version    Show version information")
	fmt.Println("  help       Show this help message")
	fmt.Println()
	fmt.Println("Examples:")
	fmt.Println("  taskmaster add \"Buy groceries\" -priority high")
	fmt.Println("  taskmaster add \"Write report\" -desc \"Q4 summary\" -priority medium")
	fmt.Println("  taskmaster list")
	fmt.Println("  taskmaster list -status pending")
	fmt.Println("  taskmaster list -priority high")
	fmt.Println("  taskmaster complete 1")
	fmt.Println("  taskmaster delete 2")
}
