package cli

import (
	"fmt"
	"os"
	"text/tabwriter"
	"time"

	"github.com/fatih/color"
	"taskmaster/internal/models"
	"taskmaster/internal/storage"
)

// ListTasks handles the 'list' command
func ListTasks(store storage.Storage, filter *storage.TaskFilter) error {
	tasks, err := store.ListTasks(filter)
	if err != nil {
		return fmt.Errorf("failed to list tasks: %w", err)
	}

	if len(tasks) == 0 {
		color.Yellow("No tasks found.")
		return nil
	}

	// Create tabwriter for aligned output
	w := tabwriter.NewWriter(os.Stdout, 0, 0, 2, ' ', 0)

	// Print header
	bold := color.New(color.Bold)
	bold.Fprintf(w, "ID\tSTATUS\tPRIORITY\tTITLE\tCREATED\n")
	fmt.Fprintf(w, "──\t──────\t────────\t─────\t───────\n")

	// Print tasks
	for _, task := range tasks {
		status := "[ ]"
		statusColor := color.New(color.FgYellow)
		if task.Completed {
			status = "[✓]"
			statusColor = color.New(color.FgGreen)
		}

		priorityColor := getPriorityColor(task.Priority)

		fmt.Fprintf(w, "%d\t", task.ID)
		statusColor.Fprintf(w, "%s\t", status)
		priorityColor.Fprintf(w, "%s\t", task.Priority)
		fmt.Fprintf(w, "%s\t", task.Title)
		fmt.Fprintf(w, "%s\n", formatTime(task.CreatedAt))
	}

	w.Flush()

	// Print summary
	completed := 0
	for _, task := range tasks {
		if task.Completed {
			completed++
		}
	}

	fmt.Printf("\nTotal: %d tasks (%d completed, %d pending)\n",
		len(tasks), completed, len(tasks)-completed)

	return nil
}

// getPriorityColor returns the color for a given priority level
func getPriorityColor(p models.Priority) *color.Color {
	switch p {
	case models.PriorityCritical:
		return color.New(color.FgRed, color.Bold)
	case models.PriorityHigh:
		return color.New(color.FgRed)
	case models.PriorityMedium:
		return color.New(color.FgYellow)
	default:
		return color.New(color.FgCyan)
	}
}

// formatTime formats a time.Time into a human-readable string
func formatTime(t time.Time) string {
	diff := time.Since(t)

	if diff < time.Minute {
		return "just now"
	}
	if diff < time.Hour {
		return fmt.Sprintf("%d min ago", int(diff.Minutes()))
	}
	if diff < 24*time.Hour {
		return fmt.Sprintf("%d hours ago", int(diff.Hours()))
	}
	if diff < 7*24*time.Hour {
		return fmt.Sprintf("%d days ago", int(diff.Hours()/24))
	}

	return t.Format("Jan 2, 2006")
}
