package models

import (
	"errors"
	"time"
)

// Priority represents task priority levels
type Priority int

const (
	PriorityLow Priority = iota
	PriorityMedium
	PriorityHigh
	PriorityCritical
)

// String returns the string representation of priority
func (p Priority) String() string {
	switch p {
	case PriorityLow:
		return "Low"
	case PriorityMedium:
		return "Medium"
	case PriorityHigh:
		return "High"
	case PriorityCritical:
		return "Critical"
	default:
		return "Unknown"
	}
}

// Task represents a single task
type Task struct {
	ID          int
	Title       string
	Description string
	Priority    Priority
	Completed   bool
	CreatedAt   time.Time
	CompletedAt *time.Time
}

// Validate checks if the task is valid
func (t *Task) Validate() error {
	if t.Title == "" {
		return errors.New("task title cannot be empty")
	}
	if len(t.Title) > 200 {
		return errors.New("task title too long (max 200 characters)")
	}
	if t.Priority < PriorityLow || t.Priority > PriorityCritical {
		return errors.New("invalid priority")
	}
	return nil
}

// MarkComplete marks the task as completed
func (t *Task) MarkComplete() {
	t.Completed = true
	now := time.Now()
	t.CompletedAt = &now
}
