package storage

import (
	"database/sql"
	"fmt"

	_ "github.com/mattn/go-sqlite3"
	"taskmaster/internal/models"
)

// SQLiteStorage implements the Storage interface using SQLite
type SQLiteStorage struct {
	db *sql.DB
}

// NewSQLiteStorage creates a new SQLite storage backend
func NewSQLiteStorage(dbPath string) (*SQLiteStorage, error) {
	db, err := sql.Open("sqlite3", dbPath)
	if err != nil {
		return nil, fmt.Errorf("failed to open database: %w", err)
	}

	storage := &SQLiteStorage{db: db}
	if err := storage.migrate(); err != nil {
		return nil, fmt.Errorf("failed to migrate database: %w", err)
	}

	return storage, nil
}

// migrate creates the database schema
func (s *SQLiteStorage) migrate() error {
	query := `
	CREATE TABLE IF NOT EXISTS tasks (
		id INTEGER PRIMARY KEY AUTOINCREMENT,
		title TEXT NOT NULL,
		description TEXT,
		priority INTEGER NOT NULL,
		completed BOOLEAN NOT NULL DEFAULT 0,
		created_at DATETIME NOT NULL,
		completed_at DATETIME
	);
	CREATE INDEX IF NOT EXISTS idx_completed ON tasks(completed);
	CREATE INDEX IF NOT EXISTS idx_priority ON tasks(priority);
	`
	_, err := s.db.Exec(query)
	return err
}

// CreateTask adds a new task to the database
func (s *SQLiteStorage) CreateTask(task *models.Task) error {
	if err := task.Validate(); err != nil {
		return fmt.Errorf("invalid task: %w", err)
	}

	query := `
		INSERT INTO tasks (title, description, priority, completed, created_at)
		VALUES (?, ?, ?, ?, ?)
	`

	result, err := s.db.Exec(
		query,
		task.Title,
		task.Description,
		task.Priority,
		task.Completed,
		task.CreatedAt,
	)
	if err != nil {
		return fmt.Errorf("failed to insert task: %w", err)
	}

	id, err := result.LastInsertId()
	if err != nil {
		return fmt.Errorf("failed to get task ID: %w", err)
	}

	task.ID = int(id)
	return nil
}

// GetTask retrieves a task by ID
func (s *SQLiteStorage) GetTask(id int) (*models.Task, error) {
	query := `
		SELECT id, title, description, priority, completed, created_at, completed_at
		FROM tasks WHERE id = ?
	`

	task := &models.Task{}
	var completedAt sql.NullTime

	err := s.db.QueryRow(query, id).Scan(
		&task.ID,
		&task.Title,
		&task.Description,
		&task.Priority,
		&task.Completed,
		&task.CreatedAt,
		&completedAt,
	)

	if err == sql.ErrNoRows {
		return nil, fmt.Errorf("task not found")
	}
	if err != nil {
		return nil, fmt.Errorf("failed to query task: %w", err)
	}

	if completedAt.Valid {
		task.CompletedAt = &completedAt.Time
	}

	return task, nil
}

// ListTasks returns all tasks matching the filter
func (s *SQLiteStorage) ListTasks(filter *TaskFilter) ([]*models.Task, error) {
	query := "SELECT id, title, description, priority, completed, created_at, completed_at FROM tasks WHERE 1=1"
	args := []interface{}{}

	if filter != nil {
		if filter.Status == "pending" {
			query += " AND completed = 0"
		} else if filter.Status == "completed" {
			query += " AND completed = 1"
		}

		if filter.Priority != nil {
			query += " AND priority = ?"
			args = append(args, *filter.Priority)
		}
	}

	query += " ORDER BY created_at DESC"

	rows, err := s.db.Query(query, args...)
	if err != nil {
		return nil, fmt.Errorf("failed to query tasks: %w", err)
	}
	defer rows.Close()

	var tasks []*models.Task
	for rows.Next() {
		task := &models.Task{}
		var completedAt sql.NullTime

		err := rows.Scan(
			&task.ID,
			&task.Title,
			&task.Description,
			&task.Priority,
			&task.Completed,
			&task.CreatedAt,
			&completedAt,
		)
		if err != nil {
			return nil, fmt.Errorf("failed to scan task: %w", err)
		}

		if completedAt.Valid {
			task.CompletedAt = &completedAt.Time
		}

		tasks = append(tasks, task)
	}

	return tasks, rows.Err()
}

// UpdateTask updates an existing task
func (s *SQLiteStorage) UpdateTask(task *models.Task) error {
	if err := task.Validate(); err != nil {
		return fmt.Errorf("invalid task: %w", err)
	}

	query := `
		UPDATE tasks
		SET title = ?, description = ?, priority = ?, completed = ?, completed_at = ?
		WHERE id = ?
	`

	var completedAt interface{}
	if task.CompletedAt != nil {
		completedAt = task.CompletedAt
	}

	_, err := s.db.Exec(
		query,
		task.Title,
		task.Description,
		task.Priority,
		task.Completed,
		completedAt,
		task.ID,
	)

	if err != nil {
		return fmt.Errorf("failed to update task: %w", err)
	}

	return nil
}

// DeleteTask removes a task by ID
func (s *SQLiteStorage) DeleteTask(id int) error {
	result, err := s.db.Exec("DELETE FROM tasks WHERE id = ?", id)
	if err != nil {
		return fmt.Errorf("failed to delete task: %w", err)
	}

	rows, err := result.RowsAffected()
	if err != nil {
		return fmt.Errorf("failed to get rows affected: %w", err)
	}

	if rows == 0 {
		return fmt.Errorf("task not found")
	}

	return nil
}

// Close closes the database connection
func (s *SQLiteStorage) Close() error {
	return s.db.Close()
}
