package storage

import (
	"os"
	"testing"
	"time"

	"taskmaster/internal/models"
)

func setupTestDB(t *testing.T) (*SQLiteStorage, func()) {
	t.Helper()

	dbPath := "test_tasks.db"
	store, err := NewSQLiteStorage(dbPath)
	if err != nil {
		t.Fatalf("failed to create test database: %v", err)
	}

	cleanup := func() {
		store.Close()
		os.Remove(dbPath)
	}

	return store, cleanup
}

func TestCreateTask(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	task := &models.Task{
		Title:     "Test Task",
		Priority:  models.PriorityHigh,
		CreatedAt: time.Now(),
	}

	err := store.CreateTask(task)
	if err != nil {
		t.Fatalf("CreateTask failed: %v", err)
	}

	if task.ID == 0 {
		t.Error("Task ID should be set after creation")
	}
}

func TestGetTask(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	original := &models.Task{
		Title:       "Test Task",
		Description: "Test Description",
		Priority:    models.PriorityMedium,
		CreatedAt:   time.Now(),
	}

	store.CreateTask(original)

	retrieved, err := store.GetTask(original.ID)
	if err != nil {
		t.Fatalf("GetTask failed: %v", err)
	}

	if retrieved.Title != original.Title {
		t.Errorf("Expected title %s, got %s", original.Title, retrieved.Title)
	}

	if retrieved.Priority != original.Priority {
		t.Errorf("Expected priority %v, got %v", original.Priority, retrieved.Priority)
	}
}

func TestListTasks(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	// Create test tasks
	tasks := []*models.Task{
		{Title: "Task 1", Priority: models.PriorityLow, CreatedAt: time.Now()},
		{Title: "Task 2", Priority: models.PriorityHigh, CreatedAt: time.Now()},
		{Title: "Task 3", Priority: models.PriorityMedium, CreatedAt: time.Now()},
	}

	for _, task := range tasks {
		store.CreateTask(task)
	}

	// Mark one as complete
	tasks[1].MarkComplete()
	store.UpdateTask(tasks[1])

	// Test listing all tasks
	all, err := store.ListTasks(&TaskFilter{Status: "all"})
	if err != nil {
		t.Fatalf("ListTasks failed: %v", err)
	}

	if len(all) != 3 {
		t.Errorf("Expected 3 tasks, got %d", len(all))
	}

	// Test filtering by status
	pending, _ := store.ListTasks(&TaskFilter{Status: "pending"})
	if len(pending) != 2 {
		t.Errorf("Expected 2 pending tasks, got %d", len(pending))
	}

	completed, _ := store.ListTasks(&TaskFilter{Status: "completed"})
	if len(completed) != 1 {
		t.Errorf("Expected 1 completed task, got %d", len(completed))
	}
}

func TestUpdateTask(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	task := &models.Task{
		Title:     "Original Title",
		Priority:  models.PriorityLow,
		CreatedAt: time.Now(),
	}

	store.CreateTask(task)

	// Update task
	task.Title = "Updated Title"
	task.Priority = models.PriorityHigh

	err := store.UpdateTask(task)
	if err != nil {
		t.Fatalf("UpdateTask failed: %v", err)
	}

	// Verify update
	retrieved, _ := store.GetTask(task.ID)
	if retrieved.Title != "Updated Title" {
		t.Error("Title was not updated")
	}
	if retrieved.Priority != models.PriorityHigh {
		t.Error("Priority was not updated")
	}
}

func TestDeleteTask(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	task := &models.Task{
		Title:     "To Be Deleted",
		Priority:  models.PriorityLow,
		CreatedAt: time.Now(),
	}

	store.CreateTask(task)

	err := store.DeleteTask(task.ID)
	if err != nil {
		t.Fatalf("DeleteTask failed: %v", err)
	}

	// Verify deletion
	_, err = store.GetTask(task.ID)
	if err == nil {
		t.Error("Expected error when getting deleted task")
	}
}

func TestTaskValidation(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	// Test empty title
	task := &models.Task{
		Title:     "",
		Priority:  models.PriorityLow,
		CreatedAt: time.Now(),
	}

	err := store.CreateTask(task)
	if err == nil {
		t.Error("Expected error for empty title")
	}

	// Test long title
	task.Title = string(make([]byte, 201))
	err = store.CreateTask(task)
	if err == nil {
		t.Error("Expected error for too long title")
	}
}

func TestPriorityFiltering(t *testing.T) {
	store, cleanup := setupTestDB(t)
	defer cleanup()

	// Create tasks with different priorities
	tasks := []*models.Task{
		{Title: "Low Priority", Priority: models.PriorityLow, CreatedAt: time.Now()},
		{Title: "High Priority", Priority: models.PriorityHigh, CreatedAt: time.Now()},
		{Title: "Critical Priority", Priority: models.PriorityCritical, CreatedAt: time.Now()},
	}

	for _, task := range tasks {
		store.CreateTask(task)
	}

	// Filter by high priority
	highPriority := models.PriorityHigh
	filtered, err := store.ListTasks(&TaskFilter{
		Status:   "all",
		Priority: &highPriority,
	})

	if err != nil {
		t.Fatalf("ListTasks with priority filter failed: %v", err)
	}

	if len(filtered) != 1 {
		t.Errorf("Expected 1 high priority task, got %d", len(filtered))
	}

	if len(filtered) > 0 && filtered[0].Priority != models.PriorityHigh {
		t.Error("Expected high priority task")
	}
}
