package main

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"
	"github.com/yourusername/etl-pipeline/internal/config"
	"github.com/yourusername/etl-pipeline/internal/extract"
	"github.com/yourusername/etl-pipeline/internal/load"
	"github.com/yourusername/etl-pipeline/internal/transform"
	"github.com/yourusername/etl-pipeline/pkg/models"
)

var (
	cfgFile      string
	verbose      bool
	extractType  string
	extractFile  string
	loadType     string
	loadFile     string
	filterExpr   string
)

var rootCmd = &cobra.Command{
	Use:   "etl",
	Short: "ETL Pipeline CLI",
	Long:  "A production-ready ETL pipeline for extracting, transforming, and loading data",
}

var pipelineCmd = &cobra.Command{
	Use:   "pipeline",
	Short: "Run full ETL pipeline",
	Long:  "Execute a complete ETL pipeline with extract, transform, and load phases",
	RunE:  runPipeline,
}

func init() {
	rootCmd.PersistentFlags().StringVar(&cfgFile, "config", "", "config file path")
	rootCmd.PersistentFlags().BoolVarP(&verbose, "verbose", "v", false, "verbose output")

	pipelineCmd.Flags().StringVar(&extractType, "extract-type", "csv", "extract type (csv, json)")
	pipelineCmd.Flags().StringVar(&extractFile, "extract-file", "", "extract file path")
	pipelineCmd.Flags().StringVar(&loadType, "load-type", "json", "load type (json, database)")
	pipelineCmd.Flags().StringVar(&loadFile, "load-file", "output.json", "load file path")
	pipelineCmd.Flags().StringVar(&filterExpr, "filter", "", "filter expression (e.g., 'quantity > 0')")

	rootCmd.AddCommand(pipelineCmd)
}

func runPipeline(cmd *cobra.Command, args []string) error {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// Handle signals for graceful shutdown
	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, os.Interrupt, syscall.SIGTERM)
	go func() {
		<-sigCh
		fmt.Println("\nShutting down gracefully...")
		cancel()
	}()

	// Load config if provided
	var cfg *config.Config
	if cfgFile != "" {
		var err error
		cfg, err = config.LoadPipelineConfig(cfgFile)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}
		if verbose {
			fmt.Printf("Loaded config: %s\n", cfg.Pipeline.Name)
		}
	}

	// Determine extract parameters
	if cfg != nil {
		extractType = cfg.Extract.Type
		extractFile = cfg.Extract.Source
	}

	if extractFile == "" {
		return fmt.Errorf("extract-file is required")
	}

	// Create extractor
	var extractor extract.Extractor
	var err error

	extractOpts := extract.Options{
		BufferSize:  1000,
		SkipInvalid: false,
	}

	if verbose {
		fmt.Printf("Extracting from %s (%s)...\n", extractFile, extractType)
	}

	switch extractType {
	case "csv":
		extractor, err = extract.NewCSVExtractor(extractFile, extractOpts)
	case "json":
		extractor, err = extract.NewJSONExtractor(extractFile, extractOpts)
	default:
		return fmt.Errorf("unknown extract type: %s", extractType)
	}

	if err != nil {
		return fmt.Errorf("failed to create extractor: %w", err)
	}
	defer extractor.Close()

	// Determine load parameters
	if cfg != nil {
		loadType = cfg.Load.Type
		loadFile = cfg.Load.Destination
	}

	// Create loader
	var loader load.Loader

	loadOpts := load.Options{
		BatchSize: 1000,
		Pretty:    true,
	}

	if verbose {
		fmt.Printf("Loading to %s (%s)...\n", loadFile, loadType)
	}

	switch loadType {
	case "json":
		loader, err = load.NewJSONLoader(loadFile, loadOpts)
	case "database":
		// Create schema for database
		schema := models.NewSchema()
		schema.AddField("date", models.TypeString)
		schema.AddField("product", models.TypeString)
		schema.AddField("quantity", models.TypeInt)
		schema.AddField("price", models.TypeFloat)

		loader, err = load.NewDatabaseLoader(loadFile, "sales", schema, loadOpts)
	default:
		return fmt.Errorf("unknown load type: %s", loadType)
	}

	if err != nil {
		return fmt.Errorf("failed to create loader: %w", err)
	}
	defer loader.Close()

	// Start extraction
	recordCh, errCh := extractor.Extract(ctx)

	// Apply transformations (optional)
	transformedCh := recordCh

	if filterExpr != "" || (cfg != nil && len(cfg.Transform.Filters) > 0) {
		filters := []string{}
		if filterExpr != "" {
			filters = append(filters, filterExpr)
		}
		if cfg != nil {
			filters = append(filters, cfg.Transform.Filters...)
		}

		if verbose && len(filters) > 0 {
			fmt.Printf("Applying %d filter(s)...\n", len(filters))
		}

		transformedCh = applyFilters(ctx, recordCh, filters)
	}

	// Load data
	if err := loader.Load(ctx, transformedCh); err != nil {
		return fmt.Errorf("load failed: %w", err)
	}

	// Check for extract errors
	select {
	case err := <-errCh:
		if err != nil {
			return fmt.Errorf("extract failed: %w", err)
		}
	default:
	}

	fmt.Println("Pipeline completed successfully!")
	return nil
}

func applyFilters(ctx context.Context, input <-chan *models.Record, filterExprs []string) <-chan *models.Record {
	output := make(chan *models.Record, 1000)

	filters := make([]*transform.Filter, len(filterExprs))
	for i, expr := range filterExprs {
		filters[i] = transform.NewFilter(expr)
	}

	go func() {
		defer close(output)

		for record := range input {
			passed := true

			for _, filter := range filters {
				match, err := filter.Apply(record)
				if err != nil || !match {
					passed = false
					break
				}
			}

			if passed {
				select {
				case output <- record:
				case <-ctx.Done():
					return
				}
			}
		}
	}()

	return output
}

func main() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n", err)
		os.Exit(1)
	}
}
