package config

import (
	"encoding/json"
	"os"
)

// Config represents the pipeline configuration
type Config struct {
	Pipeline  PipelineConfig  `json:"pipeline"`
	Extract   ExtractConfig   `json:"extract"`
	Transform TransformConfig `json:"transform"`
	Load      LoadConfig      `json:"load"`
	Execution ExecutionConfig `json:"execution"`
}

type PipelineConfig struct {
	Name        string `json:"name"`
	Description string `json:"description"`
}

type ExtractConfig struct {
	Type    string                 `json:"type"`
	Source  string                 `json:"source"`
	Options map[string]interface{} `json:"options"`
	Schema  map[string]string      `json:"schema"`
}

type TransformConfig struct {
	Filters      []string          `json:"filters"`
	Mappings     map[string]string `json:"mappings"`
	Aggregations []AggregationConfig `json:"aggregations"`
}

type AggregationConfig struct {
	Operation string   `json:"operation"`
	Field     string   `json:"field"`
	GroupBy   []string `json:"group_by"`
}

type LoadConfig struct {
	Type        string                 `json:"type"`
	Destination string                 `json:"destination"`
	Options     map[string]interface{} `json:"options"`
}

type ExecutionConfig struct {
	Concurrent bool `json:"concurrent"`
	Workers    int  `json:"workers"`
	BufferSize int  `json:"buffer_size"`
	Progress   bool `json:"progress"`
}

// LoadPipelineConfig loads configuration from file
func LoadPipelineConfig(filename string) (*Config, error) {
	file, err := os.Open(filename)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	var config Config
	if err := json.NewDecoder(file).Decode(&config); err != nil {
		return nil, err
	}

	return &config, nil
}
