package extract

import (
	"context"
	"encoding/csv"
	"fmt"
	"io"
	"os"
	"strconv"

	"github.com/yourusername/etl-pipeline/pkg/models"
)

// CSVExtractor extracts data from CSV files
type CSVExtractor struct {
	file    *os.File
	reader  *csv.Reader
	header  []string
	schema  *models.Schema
	options Options
}

// NewCSVExtractor creates a new CSV extractor
func NewCSVExtractor(filename string, options Options) (*CSVExtractor, error) {
	file, err := os.Open(filename)
	if err != nil {
		return nil, fmt.Errorf("failed to open file: %w", err)
	}

	reader := csv.NewReader(file)

	// Read header
	header, err := reader.Read()
	if err != nil {
		file.Close()
		return nil, fmt.Errorf("failed to read header: %w", err)
	}

	return &CSVExtractor{
		file:    file,
		reader:  reader,
		header:  header,
		schema:  options.Schema,
		options: options,
	}, nil
}

// Extract extracts records from CSV
func (e *CSVExtractor) Extract(ctx context.Context) (<-chan *models.Record, <-chan error) {
	recordCh := make(chan *models.Record, e.options.BufferSize)
	errCh := make(chan error, 1)

	go func() {
		defer close(recordCh)
		defer close(errCh)

		lineNum := 1 // Header is line 0

		for {
			select {
			case <-ctx.Done():
				errCh <- ctx.Err()
				return
			default:
			}

			row, err := e.reader.Read()
			if err == io.EOF {
				return
			}
			if err != nil {
				if !e.options.SkipInvalid {
					errCh <- fmt.Errorf("line %d: %w", lineNum, err)
					return
				}
				lineNum++
				continue
			}

			record, err := e.parseRow(row)
			if err != nil {
				if !e.options.SkipInvalid {
					errCh <- fmt.Errorf("line %d: %w", lineNum, err)
					return
				}
				lineNum++
				continue
			}

			// Validate against schema if provided
			if e.schema != nil {
				if err := e.schema.Validate(record); err != nil {
					if !e.options.SkipInvalid {
						errCh <- fmt.Errorf("line %d: %w", lineNum, err)
						return
					}
					lineNum++
					continue
				}
			}

			select {
			case recordCh <- record:
			case <-ctx.Done():
				errCh <- ctx.Err()
				return
			}

			lineNum++
		}
	}()

	return recordCh, errCh
}

func (e *CSVExtractor) parseRow(row []string) (*models.Record, error) {
	if len(row) != len(e.header) {
		return nil, fmt.Errorf("expected %d fields, got %d", len(e.header), len(row))
	}

	record := models.NewRecord()

	for i, fieldName := range e.header {
		value := row[i]

		// Type conversion based on schema
		if e.schema != nil {
			fieldType, ok := e.schema.Fields[fieldName]
			if ok {
				converted, err := e.convertValue(value, fieldType)
				if err != nil {
					return nil, fmt.Errorf("field %s: %w", fieldName, err)
				}
				record.Set(fieldName, converted)
				continue
			}
		}

		// Default: store as string
		record.Set(fieldName, value)
	}

	return record, nil
}

func (e *CSVExtractor) convertValue(value string, fieldType models.FieldType) (interface{}, error) {
	switch fieldType {
	case models.TypeString:
		return value, nil
	case models.TypeInt:
		return strconv.Atoi(value)
	case models.TypeFloat:
		return strconv.ParseFloat(value, 64)
	case models.TypeBool:
		return strconv.ParseBool(value)
	default:
		return value, nil
	}
}

// Close closes the CSV file
func (e *CSVExtractor) Close() error {
	return e.file.Close()
}
