package extract

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"os"

	"github.com/yourusername/etl-pipeline/pkg/models"
)

// JSONExtractor extracts data from JSON files
type JSONExtractor struct {
	file    *os.File
	decoder *json.Decoder
	schema  *models.Schema
	options Options
	isArray bool
}

// NewJSONExtractor creates a new JSON extractor
func NewJSONExtractor(filename string, options Options) (*JSONExtractor, error) {
	file, err := os.Open(filename)
	if err != nil {
		return nil, fmt.Errorf("failed to open file: %w", err)
	}

	decoder := json.NewDecoder(file)

	// Check if root is array
	token, err := decoder.Token()
	if err != nil {
		file.Close()
		return nil, fmt.Errorf("failed to read first token: %w", err)
	}

	_, isArray := token.(json.Delim)

	return &JSONExtractor{
		file:    file,
		decoder: decoder,
		schema:  options.Schema,
		options: options,
		isArray: isArray && token == '[',
	}, nil
}

// Extract extracts records from JSON
func (e *JSONExtractor) Extract(ctx context.Context) (<-chan *models.Record, <-chan error) {
	recordCh := make(chan *models.Record, e.options.BufferSize)
	errCh := make(chan error, 1)

	go func() {
		defer close(recordCh)
		defer close(errCh)

		if e.isArray {
			e.extractArray(ctx, recordCh, errCh)
		} else {
			e.extractObject(ctx, recordCh, errCh)
		}
	}()

	return recordCh, errCh
}

func (e *JSONExtractor) extractArray(ctx context.Context, recordCh chan<- *models.Record, errCh chan<- error) {
	for e.decoder.More() {
		select {
		case <-ctx.Done():
			errCh <- ctx.Err()
			return
		default:
		}

		record := models.NewRecord()
		if err := e.decoder.Decode(&record.Fields); err != nil {
			if !e.options.SkipInvalid {
				errCh <- err
				return
			}
			continue
		}

		// Validate against schema
		if e.schema != nil {
			if err := e.schema.Validate(record); err != nil {
				if !e.options.SkipInvalid {
					errCh <- err
					return
				}
				continue
			}
		}

		select {
		case recordCh <- record:
		case <-ctx.Done():
			errCh <- ctx.Err()
			return
		}
	}
}

func (e *JSONExtractor) extractObject(ctx context.Context, recordCh chan<- *models.Record, errCh chan<- error) {
	// For single object, reset file and read
	e.file.Seek(0, 0)
	e.decoder = json.NewDecoder(e.file)

	record := models.NewRecord()
	if err := e.decoder.Decode(&record.Fields); err != nil && err != io.EOF {
		errCh <- err
		return
	}

	// Validate against schema
	if e.schema != nil {
		if err := e.schema.Validate(record); err != nil {
			errCh <- err
			return
		}
	}

	select {
	case recordCh <- record:
	case <-ctx.Done():
		errCh <- ctx.Err()
	}
}

// Close closes the JSON file
func (e *JSONExtractor) Close() error {
	return e.file.Close()
}
