package load

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	_ "github.com/mattn/go-sqlite3"
	"github.com/yourusername/etl-pipeline/pkg/models"
)

// DatabaseLoader loads data to SQLite database
type DatabaseLoader struct {
	db        *sql.DB
	tableName string
	options   Options
	schema    *models.Schema
	stmt      *sql.Stmt
}

// NewDatabaseLoader creates a new database loader
func NewDatabaseLoader(dbPath, tableName string, schema *models.Schema, options Options) (*DatabaseLoader, error) {
	db, err := sql.Open("sqlite3", dbPath)
	if err != nil {
		return nil, fmt.Errorf("failed to open database: %w", err)
	}

	loader := &DatabaseLoader{
		db:        db,
		tableName: tableName,
		options:   options,
		schema:    schema,
	}

	// Create table if needed
	if err := loader.createTable(); err != nil {
		db.Close()
		return nil, err
	}

	// Prepare insert statement
	if err := loader.prepareInsert(); err != nil {
		db.Close()
		return nil, err
	}

	return loader, nil
}

func (l *DatabaseLoader) createTable() error {
	if l.schema == nil {
		return fmt.Errorf("schema required for database loader")
	}

	var columns []string
	for fieldName, fieldType := range l.schema.Fields {
		sqlType := l.toSQLType(fieldType)
		columns = append(columns, fmt.Sprintf("%s %s", fieldName, sqlType))
	}

	query := fmt.Sprintf(
		"CREATE TABLE IF NOT EXISTS %s (%s)",
		l.tableName,
		strings.Join(columns, ", "),
	)

	_, err := l.db.Exec(query)
	return err
}

func (l *DatabaseLoader) toSQLType(fieldType models.FieldType) string {
	switch fieldType {
	case models.TypeString:
		return "TEXT"
	case models.TypeInt:
		return "INTEGER"
	case models.TypeFloat:
		return "REAL"
	case models.TypeBool:
		return "INTEGER"
	default:
		return "TEXT"
	}
}

func (l *DatabaseLoader) prepareInsert() error {
	if l.schema == nil {
		return fmt.Errorf("schema required")
	}

	var fields []string
	var placeholders []string

	for fieldName := range l.schema.Fields {
		fields = append(fields, fieldName)
		placeholders = append(placeholders, "?")
	}

	query := fmt.Sprintf(
		"INSERT INTO %s (%s) VALUES (%s)",
		l.tableName,
		strings.Join(fields, ", "),
		strings.Join(placeholders, ", "),
	)

	stmt, err := l.db.Prepare(query)
	if err != nil {
		return fmt.Errorf("failed to prepare statement: %w", err)
	}

	l.stmt = stmt
	return nil
}

// Load loads records to database
func (l *DatabaseLoader) Load(ctx context.Context, records <-chan *models.Record) error {
	tx, err := l.db.BeginTx(ctx, nil)
	if err != nil {
		return fmt.Errorf("failed to begin transaction: %w", err)
	}
	defer tx.Rollback()

	txStmt := tx.Stmt(l.stmt)

	batch := make([]*models.Record, 0, l.options.BatchSize)

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case record, ok := <-records:
			if !ok {
				// Process remaining batch
				if len(batch) > 0 {
					if err := l.insertBatch(txStmt, batch); err != nil {
						return err
					}
				}
				return tx.Commit()
			}

			batch = append(batch, record)

			if len(batch) >= l.options.BatchSize {
				if err := l.insertBatch(txStmt, batch); err != nil {
					return err
				}
				batch = batch[:0]
			}
		}
	}
}

func (l *DatabaseLoader) insertBatch(stmt *sql.Stmt, batch []*models.Record) error {
	for _, record := range batch {
		var values []interface{}

		for fieldName := range l.schema.Fields {
			val, ok := record.Get(fieldName)
			if !ok {
				values = append(values, nil)
			} else {
				values = append(values, val)
			}
		}

		if _, err := stmt.Exec(values...); err != nil {
			return fmt.Errorf("failed to insert record: %w", err)
		}
	}

	return nil
}

// Close closes the database connection
func (l *DatabaseLoader) Close() error {
	if l.stmt != nil {
		l.stmt.Close()
	}
	return l.db.Close()
}
