package load

import (
	"context"
	"encoding/json"
	"fmt"
	"os"

	"github.com/yourusername/etl-pipeline/pkg/models"
)

// JSONLoader loads data to JSON files
type JSONLoader struct {
	file    *os.File
	encoder *json.Encoder
	options Options
}

// NewJSONLoader creates a new JSON loader
func NewJSONLoader(filename string, options Options) (*JSONLoader, error) {
	file, err := os.Create(filename)
	if err != nil {
		return nil, fmt.Errorf("failed to create file: %w", err)
	}

	encoder := json.NewEncoder(file)
	if options.Pretty {
		encoder.SetIndent("", "  ")
	}

	return &JSONLoader{
		file:    file,
		encoder: encoder,
		options: options,
	}, nil
}

// Load loads records to JSON
func (l *JSONLoader) Load(ctx context.Context, records <-chan *models.Record) error {
	// Write opening bracket
	if _, err := l.file.WriteString("[\n"); err != nil {
		return err
	}

	first := true

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case record, ok := <-records:
			if !ok {
				// Channel closed, finalize
				if _, err := l.file.WriteString("\n]"); err != nil {
					return err
				}
				return nil
			}

			if !first {
				if _, err := l.file.WriteString(",\n"); err != nil {
					return err
				}
			}
			first = false

			// Encode record
			data, err := json.MarshalIndent(record.Fields, "  ", "  ")
			if err != nil {
				return fmt.Errorf("failed to marshal record: %w", err)
			}

			if _, err := l.file.Write(data); err != nil {
				return err
			}
		}
	}
}

// Close closes the JSON file
func (l *JSONLoader) Close() error {
	return l.file.Close()
}
