package transform

import (
	"testing"

	"github.com/yourusername/etl-pipeline/pkg/models"
)

func TestFilter_Apply(t *testing.T) {
	tests := []struct {
		name       string
		expression string
		record     *models.Record
		expected   bool
		hasError   bool
	}{
		{
			name:       "integer greater than",
			expression: "age > 18",
			record: func() *models.Record {
				r := models.NewRecord()
				r.Set("age", 25)
				return r
			}(),
			expected: true,
			hasError: false,
		},
		{
			name:       "integer less than",
			expression: "age < 18",
			record: func() *models.Record {
				r := models.NewRecord()
				r.Set("age", 25)
				return r
			}(),
			expected: false,
			hasError: false,
		},
		{
			name:       "string equality",
			expression: "status == active",
			record: func() *models.Record {
				r := models.NewRecord()
				r.Set("status", "active")
				return r
			}(),
			expected: true,
			hasError: false,
		},
		{
			name:       "string inequality",
			expression: "status != inactive",
			record: func() *models.Record {
				r := models.NewRecord()
				r.Set("status", "active")
				return r
			}(),
			expected: true,
			hasError: false,
		},
		{
			name:       "float comparison",
			expression: "price <= 100.0",
			record: func() *models.Record {
				r := models.NewRecord()
				r.Set("price", 95.5)
				return r
			}(),
			expected: true,
			hasError: false,
		},
		{
			name:       "missing field",
			expression: "missing > 10",
			record:     models.NewRecord(),
			expected:   false,
			hasError:   true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			filter := NewFilter(tt.expression)
			result, err := filter.Apply(tt.record)

			if tt.hasError {
				if err == nil {
					t.Error("Expected error but got none")
				}
				return
			}

			if err != nil {
				t.Errorf("Unexpected error: %v", err)
				return
			}

			if result != tt.expected {
				t.Errorf("Expected %v, got %v", tt.expected, result)
			}
		})
	}
}

func TestFilter_ComplexExpressions(t *testing.T) {
	record := models.NewRecord()
	record.Set("quantity", 15)
	record.Set("price", 29.99)
	record.Set("status", "active")

	tests := []struct {
		expression string
		expected   bool
	}{
		{"quantity >= 10", true},
		{"quantity > 20", false},
		{"price < 30", true},
		{"price > 50", false},
		{"status == active", true},
		{"status != inactive", true},
	}

	for _, tt := range tests {
		filter := NewFilter(tt.expression)
		result, err := filter.Apply(record)

		if err != nil {
			t.Errorf("Unexpected error for %s: %v", tt.expression, err)
			continue
		}

		if result != tt.expected {
			t.Errorf("Expression %s: expected %v, got %v", tt.expression, tt.expected, result)
		}
	}
}
