package models

import (
	"encoding/json"
	"fmt"
	"strconv"
)

// Record represents a single data record with dynamic fields
type Record struct {
	Fields map[string]interface{}
}

// NewRecord creates a new record
func NewRecord() *Record {
	return &Record{
		Fields: make(map[string]interface{}),
	}
}

// Get retrieves a field value
func (r *Record) Get(key string) (interface{}, bool) {
	val, ok := r.Fields[key]
	return val, ok
}

// Set sets a field value
func (r *Record) Set(key string, value interface{}) {
	r.Fields[key] = value
}

// GetString retrieves a string field
func (r *Record) GetString(key string) (string, error) {
	val, ok := r.Get(key)
	if !ok {
		return "", fmt.Errorf("field %s not found", key)
	}

	switch v := val.(type) {
	case string:
		return v, nil
	default:
		return fmt.Sprintf("%v", v), nil
	}
}

// GetInt retrieves an integer field
func (r *Record) GetInt(key string) (int, error) {
	val, ok := r.Get(key)
	if !ok {
		return 0, fmt.Errorf("field %s not found", key)
	}

	switch v := val.(type) {
	case int:
		return v, nil
	case int64:
		return int(v), nil
	case float64:
		return int(v), nil
	case string:
		return strconv.Atoi(v)
	default:
		return 0, fmt.Errorf("cannot convert %T to int", v)
	}
}

// GetFloat retrieves a float field
func (r *Record) GetFloat(key string) (float64, error) {
	val, ok := r.Get(key)
	if !ok {
		return 0, fmt.Errorf("field %s not found", key)
	}

	switch v := val.(type) {
	case float64:
		return v, nil
	case int:
		return float64(v), nil
	case int64:
		return float64(v), nil
	case string:
		return strconv.ParseFloat(v, 64)
	default:
		return 0, fmt.Errorf("cannot convert %T to float64", v)
	}
}

// MarshalJSON implements json.Marshaler
func (r *Record) MarshalJSON() ([]byte, error) {
	return json.Marshal(r.Fields)
}

// UnmarshalJSON implements json.Unmarshaler
func (r *Record) UnmarshalJSON(data []byte) error {
	return json.Unmarshal(data, &r.Fields)
}
